/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user.model;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonProperty;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.user.data.UserRequestData;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * User data shared by {@link UserDto} and {@link UserRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractUser implements HasCustomFields, HasInitializedData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String uuid;
	private String pictureUrl;
	private String profileUrl;
	private String password;
	private String confirmPassword;
	private UserDetailsDto userDetails;
	private List<CustomFieldValueDto> customFields;
	private Boolean isMember;

	/**
	 * The Unique User ID of the User. This parameter uniquely identifies this
	 * user in the system and is used in a number of user related API calls.
	 * You should use the value you currently use and store to identify unique
	 * users in your system as this will facilitate the retrieval of that user's
	 * information. A good example would be the use of an email: user@acme.com.
	 * If you don't like the idea of using the email in the API URIs, then you
	 * could encode the email, for example: TWFuIGlzIGRpc3@acme.com.
	 * @return
	 */
	public String getUuid() { return uuid; }

	public void setUuid( String uuid ) {
		initializedFields.add( "uuid" );
		this.uuid = uuid;
	}

	/**
	 * The URL of the user's profile picture. This should be a fully qualified
	 * URL, for example: http://www.acme.com/user.png.
	 * @return
	 */
	public String getPictureUrl() { return pictureUrl; }
	public void setPictureUrl( String pictureUrl ) {
		initializedFields.add( "pictureUrl" );
		this.pictureUrl = pictureUrl;
	}

	/**
	 * The URL of the user's profile page. This should be a fully qualified
	 * URL, for example: http://www.acme.com/user_id.
	 * @return
	 */
	public String getProfileUrl() { return profileUrl; }
	public void setProfileUrl( String profileUrl ) {
		initializedFields.add( "profileUrl" );
		this.profileUrl = profileUrl;
	}

	/**
	 * The user's details. This includes email, first name, last name, etc.
	 * @return
	 */
	public UserDetailsDto getDetails() { return userDetails; }
	public void setDetails( UserDetailsDto userDetails ) {
		initializedFields.add( "details" );
		this.userDetails = userDetails;
	}

	/**
	 * Custom fields defined for the user. These can contain additional
	 * data such as phone number, address, or any kind of information
	 * you would like to store which isn't a standard field of the user.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * User's password for the authentication process.  {@see DatabaseAuthenticationDto}
	 * @return
	 */
	public String getPassword() { return password; }
	public void setPassword( String pwd ) {
		initializedFields.add( "password" );
		password = pwd;
	}

	/**
	 * User's password confirmation for the the authentication process. {@see DatabaseAuthenticationDto}
	 * @return
	 */
	public String getConfirmPassword() { return confirmPassword; }
	public void setConfirmPassword( String cfmPwd ) {
		initializedFields.add( "confirmPassword" );
		confirmPassword = cfmPwd;
	}

	/**
	 * Whether the user is taking part in the rewards program. By default
	 * all user's are active members of the rewards program. However, this can be set
	 * to false and the users can be explicitly asked to take part in the rewards program.
	 * If the user is not a part of the rewards program, then activities will return an
	 * exception when being tracked.
	 * @return
	 */
	@JsonProperty( "isMember" )
	public Boolean isMember() { return isMember; }
	public void setIsMember( Boolean isMember ) {
		initializedFields.add( "isMember" );
		this.isMember = isMember;
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
