/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.user.data.UserRequestData;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.service.user.param.UserCreateParams;
import io.mambo.sdk.service.user.param.UserDeleteParams;
import io.mambo.sdk.service.user.param.UserGetByIdParams;
import io.mambo.sdk.service.user.param.UserGetListParams;
import io.mambo.sdk.service.user.param.UserGetParams;
import io.mambo.sdk.service.user.param.UserResetParams;
import io.mambo.sdk.service.user.param.UserSearchParams;
import io.mambo.sdk.service.user.param.UserUpdateParams;
import io.mambo.sdk.service.user.param.UserUploadParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersService extends AbstractService
{
	private static final String USER_URI = "/v1/users/{userId}";
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_SEARCH_URI = USER_SITE_URI + "/search";
	private static final String USER_IMAGE_URI = USER_UUID_URI + "/image";
	private static final String USER_RESET_URI = USER_UUID_URI + "/reset";

	private final UsersPointService pointService;
	private final UsersRewardService rewardService;
	private final UsersCouponService couponService;
	private final UsersKpiService kpiService;
	private final UsersLeaderboardService leaderboardService;
	private final UsersActivityService activityService;
	private final UsersNotificationService notificationService;
	private final UsersBlacklistService blacklistService;
	private final UsersCustomFieldService customFieldService;


	public UsersService( ApiRequestAdapter apiClient )
	{
		super( apiClient );
		pointService = new UsersPointService( apiClient );
		rewardService = new UsersRewardService( apiClient );
		couponService = new UsersCouponService( apiClient );
		kpiService = new UsersKpiService( apiClient );
		leaderboardService = new UsersLeaderboardService( apiClient );
		activityService = new UsersActivityService( apiClient );
		notificationService = new UsersNotificationService( apiClient );
		blacklistService = new UsersBlacklistService( apiClient );
		customFieldService = new UsersCustomFieldService( apiClient );
	}


	/**
	 * Get a user
	 *
	 * @param siteUrl
	 *            The site URL from which to retrieve the user
	 * @param uuid
	 *            The unique user ID of the user to retrieve
	 * @return
	 */
	public UserDto get( String siteUrl, String uuid )
	{
		return get( UserGetParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Get a user by their UUID and site URL
	 *
	 * @param params
	 *            The parameters required to retrieve the user
	 * @return
	 */
	public UserDto get( UserGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a user by their UUID and site URL
	 *
	 * @param params
	 *            The parameters required to retrieve the user
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto get( UserGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_UUID_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user by their ID
	 *
	 * @param userId
	 *            The ID of the user to retrieve
	 * @return
	 */
	public UserDto getById( String userId )
	{
		return getById( UserGetByIdParams.builder()
			.userId( userId )
			.build() );
	}


	/**
	 * Get a user by their ID
	 *
	 * @param params
	 *            The parameters required to retrieve the user
	 * @return
	 */
	public UserDto getById( UserGetByIdParams params )
	{
		return getById( params, RequestOptions.create() );
	}


	/**
	 * Get a user by their ID
	 *
	 * @param params
	 *            The parameters required to retrieve the user
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto getById( UserGetByIdParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Create a new user
	 *
	 * @param params
	 *            The parameters required to create the user
	 * @param data
	 *            The user request data
	 * @return
	 */
	public UserDto create( UserCreateParams params, UserRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * Create a new user
	 *
	 * @param siteUrl
	 *            The URL of the site in which to create the user
	 * @param data
	 *            The user request data
	 * @return
	 */
	public UserDto create( String siteUrl, UserRequestData data )
	{
		return create( UserCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * Create a new user
	 *
	 * @param siteUrl
	 *            The URL of the site in which to create the user
	 * @param data
	 *            The user request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto create( String siteUrl, UserRequestData data, RequestOptions requestOptions )
	{
		return create( UserCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * Create a new user
	 *
	 * @param params
	 *            The parameters required to create the user
	 * @param data
	 *            The user request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto create(
		UserCreateParams params,
		UserRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_SITE_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Update an existing user
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique ID of the user to update
	 * @param data
	 *            The data with which to update the specified user object
	 * @return
	 */
	public UserDto update( String siteUrl, String uuid, UserRequestData data )
	{
		return update( UserUpdateParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), data );
	}


	/**
	 * Update an existing user
	 *
	 * @param params
	 *            The parameters required to update the user
	 * @param data
	 *            The data with which to update the specified user object
	 * @return
	 */
	public UserDto update( UserUpdateParams params, UserRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing user
	 *
	 * @param params
	 *            The parameters required to update the user
	 * @param data
	 *            The data with which to update the specified user object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto update( UserUpdateParams params, UserRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_UUID_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Upload an image for the user
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique ID of the user to update
	 * @param image
	 *            The image to upload for the user
	 * @return
	 */
	public UserDto uploadImage( String siteUrl, String uuid, File image )
	{
		return uploadImage( UserUploadParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), image );
	}


	/**
	 * Upload an image for the user
	 *
	 * @param params
	 *            The parameters required to upload an image for the user
	 * @param image
	 *            The image to upload for the user
	 * @return
	 */
	public UserDto uploadImage( UserUploadParams params, File image )
	{
		return uploadImage( params, image, RequestOptions.create() );
	}


	/**
	 * Upload an image for the user
	 *
	 * @param params
	 *            The parameters required to upload an image for the user
	 * @param image
	 *            The image to upload for the user
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto uploadImage( UserUploadParams params, File image, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_IMAGE_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image ) )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Delete a user
	 *
	 * @param siteUrl
	 *            The site URL from which to delete the user
	 * @param uuid
	 *            The unique ID of the user to delete
	 * @return
	 */
	public Status delete( String siteUrl, String uuid )
	{
		return delete( UserDeleteParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Delete a user
	 *
	 * @param params
	 *            The parameters required to delete the user
	 * @return
	 */
	public Status delete( UserDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a user
	 *
	 * @param params
	 *            The parameters required to delete the user
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( UserDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_UUID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Reset a user. This will clear all the data associated to the user, except for
	 * the basic data (i.e. username, email, name, etc).
	 *
	 * @param siteUrl
	 *            The site URL from which to reset the user
	 * @param uuid
	 *            The unique ID of the user to reset
	 * @return
	 */
	public Status reset( String siteUrl, String uuid )
	{
		return reset( UserResetParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Reset a user. This will clear all the data associated to the user, except for
	 * the basic data (i.e. username, email, name, etc).
	 *
	 * @param params
	 *            The parameters required to reset the user
	 * @return
	 */
	public Status reset( UserResetParams params )
	{
		return reset( params, RequestOptions.create() );
	}


	/**
	 * Reset a user. This will clear all the data associated to the user, except for
	 * the basic data (i.e. username, email, name, etc).
	 *
	 * @param params
	 *            The parameters required to reset the user
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status reset( UserResetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_RESET_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a list of users
	 *
	 * @param siteUrl
	 *            The site containing the users
	 * @return
	 */
	public List<UserDto> getUsers( String siteUrl )
	{
		return getUsers( UserGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of users
	 *
	 * @param params
	 *            The parameters required to retrieve a list of users
	 * @return
	 */
	public List<UserDto> getUsers( UserGetListParams params )
	{
		return getUsers( params, RequestOptions.create() );
	}


	/**
	 * Get a list of users
	 *
	 * @param params
	 *            The parameters required to retrieve a list of users
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserDto> getUsers( UserGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_SITE_URI )
			.responseClass( UserDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Search for users
	 *
	 * @param siteUrl
	 *            The site in which to search the users
	 * @param searchQuery
	 *            The search query to use
	 * @return
	 */
	public List<UserDto> searchUsers( String siteUrl, String searchQuery )
	{
		return searchUsers( UserSearchParams.builder()
			.siteUrl( siteUrl )
			.searchQuery( searchQuery )
			.build(), RequestOptions.create() );
	}


	/**
	 * Search for users
	 *
	 * @param params
	 *            The parameters required to search for users
	 * @return
	 */
	public List<UserDto> searchUsers( UserSearchParams params )
	{
		return searchUsers( params, RequestOptions.create() );
	}


	/**
	 * Search for users
	 *
	 * @param params
	 *            The parameters required to search for users
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserDto> searchUsers( UserSearchParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_SEARCH_URI )
			.responseClass( UserDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Methods related to points
	 */
	public UsersPointService points()
	{
		return pointService;
	}


	/**
	 * Methods related to rewards
	 */
	public UsersRewardService rewards()
	{
		return rewardService;
	}


	/**
	 * Methods related to coupons
	 */
	public UsersCouponService coupons()
	{
		return couponService;
	}


	/**
	 * Methods related to KPIs
	 */
	public UsersKpiService kpis()
	{
		return kpiService;
	}


	/**
	 * Methods related to leaderboards
	 */
	public UsersLeaderboardService leaderboards()
	{
		return leaderboardService;
	}


	/**
	 * Methods related to activities
	 */
	public UsersActivityService activities()
	{
		return activityService;
	}


	/**
	 * Methods related to notifications
	 */
	public UsersNotificationService notifications()
	{
		return notificationService;
	}


	/**
	 * Methods related to blacklisting
	 */
	public UsersBlacklistService blacklist()
	{
		return blacklistService;
	}


	/**
	 * Methods related to custom fields
	 */
	public UsersCustomFieldService customFields()
	{
		return customFieldService;
	}


	/**
	 * Return an empty {@link UserRequestData} object
	 *
	 * @return
	 */
	public UserRequestData newUserRequestData()
	{
		return new UserRequestData();
	}
}