/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.user.model.UserAwardedRewardDto;
import io.mambo.sdk.service.user.model.UserRewardsDto;
import io.mambo.sdk.service.user.param.reward.UserGetAchievementsParams;
import io.mambo.sdk.service.user.param.reward.UserGetAllAchievementsParams;
import io.mambo.sdk.service.user.param.reward.UserGetAllGiftsParams;
import io.mambo.sdk.service.user.param.reward.UserGetAllLevelsParams;
import io.mambo.sdk.service.user.param.reward.UserGetAllMissionsParams;
import io.mambo.sdk.service.user.param.reward.UserGetAllRewardsParams;
import io.mambo.sdk.service.user.param.reward.UserGetGiftsParams;
import io.mambo.sdk.service.user.param.reward.UserGetLevelsParams;
import io.mambo.sdk.service.user.param.reward.UserGetMissionsParams;
import io.mambo.sdk.service.user.param.reward.UserGetRewardsParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersRewardService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_REWARDS_URI = USER_UUID_URI + "/rewards";
	private static final String USER_ALL_REWARDS_URI = USER_UUID_URI + "/rewards/all";
	private static final String USER_LEVELS_URI = USER_UUID_URI + "/rewards/levels";
	private static final String USER_ALL_LEVELS_URI = USER_UUID_URI + "/rewards/levels/all";
	private static final String USER_ACHIEVEMENTS_URI = USER_UUID_URI + "/rewards/achievements";
	private static final String USER_ALL_ACHIEVEMENTS_URI = USER_UUID_URI + "/rewards/achievements/all";
	private static final String USER_MISSIONS_URI = USER_UUID_URI + "/rewards/missions";
	private static final String USER_ALL_MISSIONS_URI = USER_UUID_URI + "/rewards/missions/all";
	private static final String USER_GIFTS_URI = USER_UUID_URI + "/rewards/gifts";
	private static final String USER_ALL_GIFTS_URI = USER_UUID_URI + "/rewards/gifts/all";


	public UsersRewardService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Get a user's rewards
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getRewards( String siteUrl, String uuid )
	{
		return getRewards( UserGetRewardsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's rewards
	 *
	 * @param params
	 *            The parameters required to retrieve the user's rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getRewards( UserGetRewardsParams params )
	{
		return getRewards( params, RequestOptions.create() );
	}


	/**
	 * Get a user's rewards
	 *
	 * @param params
	 *            The parameters required to retrieve the user's rewards
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserAwardedRewardDto> getRewards( UserGetRewardsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_REWARDS_URI )
			.responseClass( UserAwardedRewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's rewards together with a full list of available rewards
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public UserRewardsDto getAllRewards( String siteUrl, String uuid )
	{
		return getAllRewards( UserGetAllRewardsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's rewards together with a full list of available rewards
	 *
	 * @param params
	 *            The parameters required to retrieve all rewards
	 * @return
	 */
	public UserRewardsDto getAllRewards( UserGetAllRewardsParams params )
	{
		return getAllRewards( params, RequestOptions.create() );
	}


	/**
	 * Get a user's rewards together with a full list of available rewards
	 *
	 * @param params
	 *            The parameters required to retrieve all rewards
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserRewardsDto getAllRewards( UserGetAllRewardsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_REWARDS_URI )
			.responseClass( UserRewardsDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's levels
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getLevels( String siteUrl, String uuid )
	{
		return getLevels( UserGetLevelsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's levels
	 *
	 * @param params
	 *            The parameters required to retrieve the user's levels
	 * @return
	 */
	public List<UserAwardedRewardDto> getLevels( UserGetLevelsParams params )
	{
		return getLevels( params, RequestOptions.create() );
	}


	/**
	 * Get a user's levels
	 *
	 * @param params
	 *            The parameters required to retrieve the user's levels
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserAwardedRewardDto> getLevels( UserGetLevelsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_LEVELS_URI )
			.responseClass( UserAwardedRewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's levels together with a full list of available levels
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public UserRewardsDto getAllLevels( String siteUrl, String uuid )
	{
		return getAllLevels( UserGetAllLevelsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's levels together with a full list of available levels
	 *
	 * @param params
	 *            The parameters required to retrieve all levels
	 * @return
	 */
	public UserRewardsDto getAllLevels( UserGetAllLevelsParams params )
	{
		return getAllLevels( params, RequestOptions.create() );
	}


	/**
	 * Get a user's levels together with a full list of available levels
	 *
	 * @param params
	 *            The parameters required to retrieve all levels
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserRewardsDto getAllLevels( UserGetAllLevelsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_LEVELS_URI )
			.responseClass( UserRewardsDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's achievements
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getAchievements( String siteUrl, String uuid )
	{
		return getAchievements( UserGetAchievementsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's achievements
	 *
	 * @param params
	 *            The parameters required to retrieve the user's achievements
	 * @return
	 */
	public List<UserAwardedRewardDto> getAchievements( UserGetAchievementsParams params )
	{
		return getAchievements( params, RequestOptions.create() );
	}


	/**
	 * Get a user's achievements
	 *
	 * @param params
	 *            The parameters required to retrieve the user's achievements
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserAwardedRewardDto> getAchievements( UserGetAchievementsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ACHIEVEMENTS_URI )
			.responseClass( UserAwardedRewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's achievements together with a full list of available achievements
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public UserRewardsDto getAllAchievements( String siteUrl, String uuid )
	{
		return getAllAchievements( UserGetAllAchievementsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's achievements together with a full list of available achievements
	 *
	 * @param params
	 *            The parameters required to retrieve all achievements
	 * @return
	 */
	public UserRewardsDto getAllAchievements( UserGetAllAchievementsParams params )
	{
		return getAllAchievements( params, RequestOptions.create() );
	}


	/**
	 * Get a user's achievements together with a full list of available achievements
	 *
	 * @param params
	 *            The parameters required to retrieve all achievements
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserRewardsDto getAllAchievements( UserGetAllAchievementsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_ACHIEVEMENTS_URI )
			.responseClass( UserRewardsDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's gifts
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getGifts( String siteUrl, String uuid )
	{
		return getGifts( UserGetGiftsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's gifts
	 *
	 * @param params
	 *            The parameters required to retrieve the user's gifts
	 * @return
	 */
	public List<UserAwardedRewardDto> getGifts( UserGetGiftsParams params )
	{
		return getGifts( params, RequestOptions.create() );
	}


	/**
	 * Get a user's gifts
	 *
	 * @param params
	 *            The parameters required to retrieve the user's gifts
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserAwardedRewardDto> getGifts( UserGetGiftsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_GIFTS_URI )
			.responseClass( UserAwardedRewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's gifts together with a full list of available gifts
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public UserRewardsDto getAllGifts( String siteUrl, String uuid )
	{
		return getAllGifts( UserGetAllGiftsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's gifts together with a full list of available gifts
	 *
	 * @param params
	 *            The parameters required to retrieve all gifts
	 * @return
	 */
	public UserRewardsDto getAllGifts( UserGetAllGiftsParams params )
	{
		return getAllGifts( params, RequestOptions.create() );
	}


	/**
	 * Get a user's gifts together with a full list of available gifts
	 *
	 * @param params
	 *            The parameters required to retrieve all gifts
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserRewardsDto getAllGifts( UserGetAllGiftsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_GIFTS_URI )
			.responseClass( UserRewardsDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's missions
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public List<UserAwardedRewardDto> getMissions( String siteUrl, String uuid )
	{
		return getMissions( UserGetMissionsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's missions
	 *
	 * @param params
	 *            The parameters required to retrieve the user's missions
	 * @return
	 */
	public List<UserAwardedRewardDto> getMissions( UserGetMissionsParams params )
	{
		return getMissions( params, RequestOptions.create() );
	}


	/**
	 * Get a user's missions
	 *
	 * @param params
	 *            The parameters required to retrieve the user's missions
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserAwardedRewardDto> getMissions( UserGetMissionsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_MISSIONS_URI )
			.responseClass( UserAwardedRewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's missions together with a full list of available missions
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the rewards
	 * @return
	 */
	public UserRewardsDto getAllMissions( String siteUrl, String uuid )
	{
		return getAllMissions( UserGetAllMissionsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's missions together with a full list of available missions
	 *
	 * @param params
	 *            The parameters required to retrieve all missions
	 * @return
	 */
	public UserRewardsDto getAllMissions( UserGetAllMissionsParams params )
	{
		return getAllMissions( params, RequestOptions.create() );
	}


	/**
	 * Get a user's missions together with a full list of available missions
	 *
	 * @param params
	 *            The parameters required to retrieve all missions
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserRewardsDto getAllMissions( UserGetAllMissionsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_MISSIONS_URI )
			.responseClass( UserRewardsDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}