/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.model.response.ResponseEntity;
import io.mambo.sdk.service.point.model.PointTransactionDto;
import io.mambo.sdk.service.point.model.PointWalletDto;
import io.mambo.sdk.service.user.param.point.UserGetPointTransactionsParams;
import io.mambo.sdk.service.user.param.point.UserGetPointWalletsParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersPointService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_POINT_WALLETS_URI = USER_UUID_URI + "/points/wallets";
	private static final String USER_POINT_TRANSACTIONS_URI = USER_UUID_URI + "/points/transactions/{pointId}";


	protected UsersPointService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Get user point wallets
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the wallets
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets( String siteUrl, String uuid )
	{
		return getPointWallets( UserGetPointWalletsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get user point wallets
	 *
	 * @param params
	 *            The parameters required to retrieve point wallets
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets(
		UserGetPointWalletsParams params )
	{
		return getPointWallets( params, RequestOptions.create() );
	}


	/**
	 * Get user point wallets
	 *
	 * @param params
	 *            The parameters required to retrieve point wallets
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets(
		UserGetPointWalletsParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_POINT_WALLETS_URI )
			.responseClass( PointWalletDto.class )
			.responseType( ResponseType.RESPONSE_ENTITY )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get user point transactions
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the transactions
	 * @param pointId
	 *            The point for which to retrieve the transactions
	 * @return
	 */
	public ResponseEntity<PointTransactionDto> getPointTransactions(
		String siteUrl, String uuid, String pointId )
	{
		return getPointTransactions( UserGetPointTransactionsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.pointId( pointId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get user point transactions
	 *
	 * @param params
	 *            The parameters required to retrieve point transactions
	 * @return
	 */
	public ResponseEntity<PointTransactionDto> getPointTransactions(
		UserGetPointTransactionsParams params )
	{
		return getPointTransactions( params, RequestOptions.create() );
	}


	/**
	 * Get user point transactions
	 *
	 * @param params
	 *            The parameters required to retrieve point transactions
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ResponseEntity<PointTransactionDto> getPointTransactions(
		UserGetPointTransactionsParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_POINT_TRANSACTIONS_URI )
			.responseClass( PointTransactionDto.class )
			.responseType( ResponseType.RESPONSE_ENTITY )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}
