/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.notification.model.NotificationDto;
import io.mambo.sdk.service.user.param.notification.UserClearNotificationsParams;
import io.mambo.sdk.service.user.param.notification.UserGetNotificationsParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersNotificationService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_NOTIFICATIONS_URI = USER_UUID_URI + "/notifications";
	private static final String USER_CLEAR_NOTIFICATIONS_URI = USER_UUID_URI + "/notifications/clear";


	protected UsersNotificationService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Get user notifications
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the notifications
	 * @return
	 */
	public List<NotificationDto> getNotifications( String siteUrl, String uuid )
	{
		return getNotifications( UserGetNotificationsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get user notifications
	 *
	 * @param params
	 *            The parameters required to retrieve user notifications
	 * @return
	 */
	public List<NotificationDto> getNotifications( UserGetNotificationsParams params )
	{
		return getNotifications( params, RequestOptions.create() );
	}


	/**
	 * Get user notifications
	 *
	 * @param params
	 *            The parameters required to retrieve user notifications
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<NotificationDto> getNotifications(
		UserGetNotificationsParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_NOTIFICATIONS_URI )
			.responseClass( NotificationDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Clear user notifications
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to clear the notifications
	 * @return
	 */
	public Status clearNotifications( String siteUrl, String uuid )
	{
		return clearNotifications( UserClearNotificationsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Clear user notifications
	 *
	 * @param params
	 *            The parameters required to clear user notifications
	 * @return
	 */
	public Status clearNotifications( UserClearNotificationsParams params )
	{
		return clearNotifications( params, RequestOptions.create() );
	}


	/**
	 * Clear user notifications
	 *
	 * @param params
	 *            The parameters required to clear user notifications
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status clearNotifications(
		UserClearNotificationsParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_CLEAR_NOTIFICATIONS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}