/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.leaderboard.model.LeaderboardDto;
import io.mambo.sdk.service.user.param.leaderboard.UserGetLeaderboardParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersLeaderboardService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_LEADERBOARD_URI = USER_UUID_URI + "/leaderboards/{leaderboardId}";


	protected UsersLeaderboardService( ApiRequestAdapter apiClient )
	{
		super( apiClient );
	}


	/**
	 * Get the leaderboard for a user
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the leaderboard
	 * @param leaderboardId
	 *            The ID of the leaderboard
	 * @return
	 */
	public LeaderboardDto getLeaderboard( String siteUrl, String uuid, String leaderboardId )
	{
		return getLeaderboard( UserGetLeaderboardParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.leaderboardId( leaderboardId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get the leaderboard for a user
	 *
	 * @param params
	 *            The parameters required to retrieve the user's leaderboard
	 * @return
	 */
	public LeaderboardDto getLeaderboard( UserGetLeaderboardParams params )
	{
		return getLeaderboard( params, RequestOptions.create() );
	}


	/**
	 * Get the leaderboard for a user
	 *
	 * @param params
	 *            The parameters required to retrieve the user's leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto getLeaderboard(
		UserGetLeaderboardParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_LEADERBOARD_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}
