/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.kpi.model.KpiStateDto;
import io.mambo.sdk.service.user.param.kpi.UserGetKpisParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersKpiService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_KPIS_URI = USER_UUID_URI + "/kpis";


	protected UsersKpiService( ApiRequestAdapter apiClient )
	{
		super( apiClient );
	}


	/**
	 * Get a user's KPIs
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the KPIs
	 * @return
	 */
	public List<KpiStateDto> getKpis( String siteUrl, String uuid )
	{
		return getKpis( UserGetKpisParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's KPIs
	 *
	 * @param params
	 *            The parameters required to retrieve the user's KPIs
	 * @return
	 */
	public List<KpiStateDto> getKpis( UserGetKpisParams params )
	{
		return getKpis( params, RequestOptions.create() );
	}


	/**
	 * Get a user's KPIs
	 *
	 * @param params
	 *            The parameters required to retrieve the user's KPIs
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<KpiStateDto> getKpis( UserGetKpisParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_KPIS_URI )
			.responseClass( KpiStateDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}