/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.io.File;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.service.user.param.customfield.UserCustomFieldParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersCustomFieldService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_CUSTOM_URI = USER_UUID_URI + "/custom_fields";
	private static final String USER_CUSTOM_IMAGE_URI = USER_UUID_URI + "/custom_fields/image";


	protected UsersCustomFieldService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Add custom fields to a user
	 *
	 * @param siteUrl
	 *            The site URL from which to retrieve the user
	 * @param uuid
	 *            The unique user ID of the user to retrieve
	 * @param data
	 *            The custom field data to add
	 * @return
	 */
	public UserDto addCustomFields(
		String siteUrl,
		String uuid,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.POST, UserCustomFieldParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Add custom fields to a user
	 *
	 * @param params
	 *            The parameters required to add custom fields
	 * @param data
	 *            The custom field data to add
	 * @return
	 */
	public UserDto addCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.POST, params, data, RequestOptions.create() );
	}


	/**
	 * Add custom fields to a user
	 *
	 * @param params
	 *            The parameters required to add custom fields
	 * @param data
	 *            The custom field data to add
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto addCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.POST, params, data, requestOptions );
	}


	/**
	 * Update custom fields of a user
	 *
	 * @param siteUrl
	 *            The site URL from which to retrieve the user
	 * @param uuid
	 *            The unique user ID of the user to retrieve
	 * @param data
	 *            The custom field data to update
	 * @return
	 */
	public UserDto updateCustomFields(
		String siteUrl,
		String uuid,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.PUT, UserCustomFieldParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Update custom fields of a user
	 *
	 * @param params
	 *            The parameters required to update custom fields
	 * @param data
	 *            The custom field data to update
	 * @return
	 */
	public UserDto updateCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.PUT, params, data, RequestOptions.create() );
	}


	/**
	 * Update custom fields of a user
	 *
	 * @param params
	 *            The parameters required to update custom fields
	 * @param data
	 *            The custom field data to update
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto updateCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.PUT, params, data, requestOptions );
	}


	/**
	 * Delete custom fields from a user
	 *
	 * @param siteUrl
	 *            The site URL from which to retrieve the user
	 * @param uuid
	 *            The unique user ID of the user to retrieve
	 * @param data
	 *            The custom field data to delete
	 * @return
	 */
	public UserDto deleteCustomFields(
		String siteUrl,
		String uuid,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.DELETE, UserCustomFieldParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Delete custom fields from a user
	 *
	 * @param params
	 *            The parameters required to delete custom fields
	 * @param data
	 *            The custom field data to delete
	 * @return
	 */
	public UserDto deleteCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, RequestOptions.create() );
	}


	/**
	 * Delete custom fields from a user
	 *
	 * @param params
	 *            The parameters required to delete custom fields
	 * @param data
	 *            The custom field data to delete
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto deleteCustomFields(
		UserCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, requestOptions );
	}


	/**
	 * Generic method used to manipulate custom fields
	 */
	private UserDto modCustomFields(
		HttpMethod method,
		UserCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_CUSTOM_URI )
			.responseClass( UserDto.class )
			.method( method )
			.params( params )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Add a custom field with an image
	 *
	 * @param siteUrl
	 *            The site URL from which to retrieve the user
	 * @param uuid
	 *            The unique user ID of the user to retrieve
	 * @param data
	 *            The custom field data
	 * @param image
	 *            The image to upload
	 * @return
	 */
	public UserDto addCustomField(
		String siteUrl,
		String uuid,
		CustomFieldValueDto data,
		File image )
	{
		return addCustomField( UserCustomFieldParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), data, image, RequestOptions.create() );
	}


	/**
	 * Add a custom field with an image
	 *
	 * @param params
	 *            The parameters required to add the custom field
	 * @param data
	 *            The custom field data
	 * @param image
	 *            The image to upload
	 * @return
	 */
	public UserDto addCustomField(
		UserCustomFieldParams params,
		CustomFieldValueDto data,
		File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * Add a custom field with an image
	 *
	 * @param params
	 *            The parameters required to add the custom field
	 * @param data
	 *            The custom field data
	 * @param image
	 *            The image to upload
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public UserDto addCustomField(
		UserCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_CUSTOM_IMAGE_URI )
			.responseClass( UserDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.requestData( new FileRequestData( image, data ) )
			.options( requestOptions )
			.build() );
	}
}