/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.coupon.model.CouponStateDto;
import io.mambo.sdk.service.coupon.model.CouponStatesDto;
import io.mambo.sdk.service.user.param.coupon.UserGetAllCouponsParams;
import io.mambo.sdk.service.user.param.coupon.UserGetCouponsParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersCouponService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_UUID_URI = USER_SITE_URI + "/{uuid}";
	private static final String USER_COUPONS_URI = USER_UUID_URI + "/coupons";
	private static final String USER_ALL_COUPONS_URI = USER_UUID_URI + "/coupons/all";


	protected UsersCouponService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Get a user's coupons
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the coupons
	 * @return
	 */
	public List<CouponStateDto> getCoupons( String siteUrl, String uuid )
	{
		return getCoupons( UserGetCouponsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's coupons
	 *
	 * @param params
	 *            The parameters required to retrieve the user's coupons
	 * @return
	 */
	public List<CouponStateDto> getCoupons( UserGetCouponsParams params )
	{
		return getCoupons( params, RequestOptions.create() );
	}


	/**
	 * Get a user's coupons
	 *
	 * @param params
	 *            The parameters required to retrieve the user's coupons
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<CouponStateDto> getCoupons( UserGetCouponsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_COUPONS_URI )
			.responseClass( CouponStateDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a user's coupons and a list of coupons which can be bought with points
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique user ID of the user for which to retrieve the coupons
	 * @return
	 */
	public CouponStatesDto getAllCoupons( String siteUrl, String uuid )
	{
		return getAllCoupons( UserGetAllCouponsParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a user's coupons and a list of coupons which can be bought with points
	 *
	 * @param params
	 *            The parameters required to retrieve all coupons
	 * @return
	 */
	public CouponStatesDto getAllCoupons( UserGetAllCouponsParams params )
	{
		return getAllCoupons( params, RequestOptions.create() );
	}


	/**
	 * Get a user's coupons and a list of coupons which can be bought with points
	 *
	 * @param params
	 *            The parameters required to retrieve all coupons
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CouponStatesDto getAllCoupons( UserGetAllCouponsParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_ALL_COUPONS_URI )
			.responseClass( CouponStatesDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}