/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.user;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.service.user.param.blacklist.UserBlacklistAddParams;
import io.mambo.sdk.service.user.param.blacklist.UserBlacklistGetListParams;
import io.mambo.sdk.service.user.param.blacklist.UserBlacklistRemoveParams;

/**
 * The UsersService class handles all User related requests to the Mambo API.
 */
public class UsersBlacklistService extends AbstractService
{
	private static final String USER_SITE_URI = "/v1/{siteUrl}/users";
	private static final String USER_BLACKLIST_URI = USER_SITE_URI + "/blacklist";
	private static final String USER_BLACKLIST_UUID_URI = USER_BLACKLIST_URI + "/{uuid}";


	protected UsersBlacklistService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Add a user to the blacklist. The blacklist stops users from earning points.
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique ID of the user to be added to the blacklist
	 * @return
	 */
	public Status addToBlacklist( String siteUrl, String uuid )
	{
		return addToBlacklist( UserBlacklistAddParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Add a user to the blacklist. The blacklist stops users from earning points.
	 *
	 * @param params
	 *            The parameters required to add the user to the blacklist
	 * @return
	 */
	public Status addToBlacklist( UserBlacklistAddParams params )
	{
		return addToBlacklist( params, RequestOptions.create() );
	}


	/**
	 * Add a user to the blacklist. The blacklist stops users from earning points.
	 *
	 * @param params
	 *            The parameters required to add the user to the blacklist
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status addToBlacklist( UserBlacklistAddParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_BLACKLIST_UUID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Remove a user from the blacklist
	 *
	 * @param siteUrl
	 *            The site URL to which the user belongs
	 * @param uuid
	 *            The unique ID of the user to be removed from the blacklist
	 * @return
	 */
	public Status removeFromBlacklist( String siteUrl, String uuid )
	{
		return removeFromBlacklist( UserBlacklistRemoveParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build() );
	}


	/**
	 * Remove a user from the blacklist
	 *
	 * @param params
	 *            The parameters required to remove the user from the blacklist
	 * @return
	 */
	public Status removeFromBlacklist( UserBlacklistRemoveParams params )
	{
		return removeFromBlacklist( params, RequestOptions.create() );
	}


	/**
	 * Remove a user from the blacklist
	 *
	 * @param params
	 *            The parameters required to remove the user from the blacklist
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status removeFromBlacklist( UserBlacklistRemoveParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_BLACKLIST_UUID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get the list of blacklisted users for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of blacklisted users
	 * @return
	 */
	public List<UserDto> getBlacklistedUsers( UserBlacklistGetListParams params )
	{
		return getBlacklistedUsers( params, RequestOptions.create() );
	}


	/**
	 * Get the list of blacklisted users for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of blacklisted users
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<UserDto> getBlacklistedUsers( UserBlacklistGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( USER_BLACKLIST_URI )
			.responseClass( UserDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}
}