/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.tag.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.tag.data.TagRequestData;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Tag data shared by {@link TagDto} and {@link TagRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractTag implements HasTranslation, HasInitializedData, HasCustomFields
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String tag;
	private Boolean personalization;
	private Boolean hideInWidgets;
	private Boolean withAnalytics;
	private List<CustomFieldValueDto> customFields;
	private List<TagTranslationDto> translations;

	/**
	 * The name of the tag.
	 * See the tag page in administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * This is the tag value that will be used when retrieving data from
	 * the API. This value must only contain letters, numbers and underscores.
	 * The tag value must be unique in order to identify this specific Tag.
	 * See the tags page in administration panel for more information.
	 * @return
	 */
	public String getTag() { return tag; }
	public void setTag( String tag ) {
		initializedFields.add( "tag" );
		this.tag = tag;
	}

	/**
	 * Indicates whether this tag is a personalization tag. Personalization
	 * tags can be used to auto-filter points, behaviours, rewards, etc by user. Assign to a user a
	 * personalization tag and they will only see the rewards, points, behaviours,
	 * etc which have a matching tag (or which have no personalization tags at all).
	 * If the user has no specific personalization tags, they will see all the points,
	 * behaviours, rewards, etc configured in the platform.
	 * See the tags page in administration panel for more information.
	 * @return
	 */
	public Boolean getPersonalization() { return personalization; }
	public void setPersonalization( Boolean personalization ) {
		initializedFields.add( "personalization" );
		this.personalization = personalization;
	}

	/**
	 * Whether the Tag should be shown or not.
	 * @return
	 */
	public Boolean getHideInWidgets() { return hideInWidgets; }
	public void setHideInWidgets( Boolean hideInWidgets ) {
		initializedFields.add( "hideInWidgets" );
		this.hideInWidgets = hideInWidgets;
	}

	/**
	 * This indicates whether the tag should be tracked in the analytics reports.
	 * This will allow you to filter the analytics reports by this particular tag.
	 * The analytics will keep track of activities performed by user's who have this tag.
	 * See the tag page in administration panel for more information.
	 * @return
	 */
	public Boolean getWithAnalytics() { return withAnalytics; }
	public void setWithAnalytics( Boolean withAnalytics ) {
		initializedFields.add( "withAnalytics" );
		this.withAnalytics = withAnalytics;
	}

	/**
	 * Custom fields defined for the tag. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the tag.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<TagTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<TagTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( TagTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public TagTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( TagTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
