/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
* Copyright (C) 2014-2024 Mambo Solutions Ltd.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package io.mambo.sdk.service.tag;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.tag.data.TagRequestData;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.service.tag.param.TagCloneParams;
import io.mambo.sdk.service.tag.param.TagCreateParams;
import io.mambo.sdk.service.tag.param.TagCustomFieldParams;
import io.mambo.sdk.service.tag.param.TagDeleteParams;
import io.mambo.sdk.service.tag.param.TagGetByIdParams;
import io.mambo.sdk.service.tag.param.TagGetListParams;
import io.mambo.sdk.service.tag.param.TagGetParams;
import io.mambo.sdk.service.tag.param.TagUpdateParams;

/**
 * The TagsService class handles all Tag related requests to the Mambo API.
 */
public class TagsService extends AbstractService
{
	private static final String TAGS_URI = "/v1/tags";
	private static final String TAGS_ID_URI = TAGS_URI + "/{tagId}";
	private static final String TAGS_SITE_URI = "/v1/{siteUrl}/tags";
	private static final String TAGS_SITE_TAG_URI = TAGS_SITE_URI + "/{tagValue}";
	private static final String TAGS_CLONE_URI = TAGS_ID_URI + "/clone";
	private static final String TAGS_CUSTOM_URI = TAGS_ID_URI + "/custom_fields";
	private static final String TAGS_CUSTOM_IMAGE_URI = TAGS_ID_URI + "/custom_fields/image";


	public TagsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new tag.
	 *
	 * @see TagRequestData
	 *
	 * @param siteUrl
	 *            The site to which the tag belongs to
	 * @param data
	 *            The tag request data
	 * @return The created tag
	 */
	public TagDto create( String siteUrl, TagRequestData data )
	{
		return create( TagCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * This method is used to create a new tag.
	 *
	 * @see TagRequestData
	 *
	 * @param siteUrl
	 *            The site to which the tag belongs to
	 * @param data
	 *            The tag request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The created tag
	 */
	public TagDto create( String siteUrl, TagRequestData data, RequestOptions requestOptions )
	{
		return create( TagCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new tag.
	 *
	 * @param params
	 *            The parameters required to create the tag
	 * @param data
	 *            The tag request data
	 * @return The created tag
	 */
	public TagDto create( TagCreateParams params, TagRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new tag.
	 *
	 * @param params
	 *            The parameters required to create the tag
	 * @param data
	 *            The tag request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The created tag
	 */
	public TagDto create(
		TagCreateParams params,
		TagRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_SITE_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing tag by ID.
	 *
	 * @param tagId
	 *            The ID of the tag to update
	 * @param data
	 *            The data with which to update the specified tag object
	 * @return The updated tag
	 */
	public TagDto update( String tagId, TagRequestData data )
	{
		return update( TagUpdateParams.builder()
			.tagId( tagId )
			.build(), data );
	}


	/**
	 * Update an existing tag by ID.
	 *
	 * @param params
	 *            The parameters required to update the tag
	 * @param data
	 *            The data with which to update the specified tag object
	 * @return The updated tag
	 */
	public TagDto update( TagUpdateParams params, TagRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing tag by ID.
	 *
	 * @param params
	 *            The parameters required to update the tag
	 * @param data
	 *            The data with which to update the specified tag object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The updated tag
	 */
	public TagDto update(
		TagUpdateParams params,
		TagRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_ID_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a tag
	 *
	 * @param tagId
	 *            The ID of the tag to clone
	 * @return The cloned tag
	 */
	public TagDto clone( String tagId )
	{
		return clone( TagCloneParams.builder()
			.tagId( tagId )
			.build() );
	}


	/**
	 * Clone a tag
	 *
	 * @param params
	 *            The parameters required to clone the tag
	 * @return The cloned tag
	 */
	public TagDto clone( TagCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a tag
	 *
	 * @param params
	 *            The parameters required to clone the tag
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The cloned tag
	 */
	public TagDto clone( TagCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_CLONE_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a tag by ID.
	 *
	 * @param tagId
	 *            The ID of the tag to delete
	 * @return The status of the delete operation
	 */
	public Status delete( String tagId )
	{
		return delete( TagDeleteParams.builder()
			.tagId( tagId )
			.build() );
	}


	/**
	 * Delete a tag by ID.
	 *
	 * @param params
	 *            The parameters required to delete the tag
	 * @return The status of the delete operation
	 */
	public Status delete( TagDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a tag by ID.
	 *
	 * @param params
	 *            The parameters required to delete the tag
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The status of the delete operation
	 */
	public Status delete( TagDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a tag by its site and tag value.
	 *
	 * @param siteUrl
	 *            The site to which the tag belongs to
	 * @param tagTag
	 *            The tag value to retrieve
	 * @return The requested tag
	 */
	public TagDto get( String siteUrl, String tagTag )
	{
		return get( TagGetParams.builder()
			.siteUrl( siteUrl )
			.tagTag( tagTag )
			.build() );
	}


	/**
	 * Get a tag by its site and tag value.
	 *
	 * @param params
	 *            The parameters required to retrieve the tag
	 * @return The requested tag
	 */
	public TagDto get( TagGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a tag by its site and tag value.
	 *
	 * @param params
	 *            The parameters required to retrieve the tag
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The requested tag
	 */
	public TagDto get( TagGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_SITE_TAG_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a tag by ID
	 *
	 * @param tagId
	 *            The ID of the tag to retrieve
	 * @return The requested tag
	 */
	public TagDto get( String tagId )
	{
		return get( TagGetByIdParams.builder()
			.tagId( tagId )
			.build() );
	}


	/**
	 * Get a tag by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the tag
	 * @return The requested tag
	 */
	public TagDto get( TagGetByIdParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a tag by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the tag
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The requested tag
	 */
	public TagDto get( TagGetByIdParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_ID_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of tags by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return The status of the delete operation
	 */
	public Status deleteTags( DeleteRequestData data )
	{
		return deleteTags( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of tags by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The status of the delete operation
	 */
	public Status deleteTags( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * Get the list of tags for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the tags
	 * @return List of tags for the specified site
	 */
	public List<TagDto> getTags( String siteUrl )
	{
		return getTags( TagGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get the list of tags for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of tags
	 * @return List of tags for the specified site
	 */
	public List<TagDto> getTags( TagGetListParams params )
	{
		return getTags( params, RequestOptions.create() );
	}


	/**
	 * Get the list of tags for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of tags
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of tags for the specified site
	 */
	public List<TagDto> getTags( TagGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_SITE_URI )
			.responseClass( TagDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Add custom fields to an existing tag.
	 *
	 * @param tagId
	 *            The tag to which to add the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto addCustomFields( String tagId, CustomFieldValueRequestData data )
	{
		TagCustomFieldParams params = TagCustomFieldParams.builder().tagId( tagId ).build();
		return modCustomFields( HttpMethod.POST, params, data );
	}


	/**
	 * Add custom fields to an existing tag.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto addCustomFields( TagCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.POST, params, data );
	}


	/**
	 * Update custom fields of an existing tag.
	 *
	 * @param tagId
	 *            The tag to update the custom fields of
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto updateCustomFields( String tagId, CustomFieldValueRequestData data )
	{
		TagCustomFieldParams params = TagCustomFieldParams.builder().tagId( tagId ).build();
		return modCustomFields( HttpMethod.PUT, params, data );
	}


	/**
	 * Update custom fields of an existing tag.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto updateCustomFields( TagCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.PUT, params, data );
	}


	/**
	 * Delete custom fields from an existing tag.
	 *
	 * @param tagId
	 *            The tag to delete the custom fields from
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto deleteCustomFields( String tagId, CustomFieldValueRequestData data )
	{
		TagCustomFieldParams params = TagCustomFieldParams.builder().tagId( tagId ).build();
		return modCustomFields( HttpMethod.DELETE, params, data );
	}


	/**
	 * Delete custom fields from an existing tag.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	public TagDto deleteCustomFields( TagCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.DELETE, params, data );
	}


	/**
	 * Generic method used to manipulate the custom fields
	 *
	 * @param method
	 *            The HTTP method to use (POST, PUT, DELETE)
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return The updated tag
	 */
	private TagDto modCustomFields( HttpMethod method, TagCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( method, params, data, RequestOptions.create() );
	}


	/**
	 * Generic method used to manipulate the custom fields
	 *
	 * @param method
	 *            The HTTP method to use (POST, PUT, DELETE)
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The updated tag
	 */
	private TagDto modCustomFields(
		HttpMethod method,
		TagCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_CUSTOM_URI )
			.responseClass( TagDto.class )
			.method( method )
			.params( params )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * Add a custom field with an image to an existing tag.
	 *
	 * @param tagId
	 *            The ID of the tag for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the tag
	 * @return The updated tag
	 */
	public TagDto addCustomField( String tagId, CustomFieldValueDto data, File image )
	{
		TagCustomFieldParams params = TagCustomFieldParams.builder().tagId( tagId ).build();
		return addCustomField( params, data, image );
	}


	/**
	 * Add a custom field with an image to an existing tag.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the tag
	 * @return The updated tag
	 */
	public TagDto addCustomField( TagCustomFieldParams params, CustomFieldValueDto data, File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * Add a custom field with an image to an existing tag.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the tag
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The updated tag
	 */
	public TagDto addCustomField(
		TagCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( TAGS_CUSTOM_IMAGE_URI )
			.responseClass( TagDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.requestData( new FileRequestData( image, data ) )
			.build() );
	}


	/**
	 * Return an empty {@link TagRequestData} object
	 *
	 * @return A new TagRequestData instance
	 */
	public TagRequestData newTagRequestData()
	{
		return new TagRequestData();
	}
}