/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.site.model.authentication;

import java.util.List;

import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates the data for the SAML authentication method.
 */
@ToString
@EqualsAndHashCode
public class SamlAuthenticationContextDto implements AuthenticationContextDto
{
	private transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String metadata;
	private List<String> defaultGroupIds;
	private SamlSyncUserAttributesDto syncUserAttributes;

	/**
	 * The type of authentication context: saml
	 *
	 * @return
	 */
	@Override
	public AuthType getAuthType()
	{
		return AuthType.SAML;
	}

	/**
	 * The type of authentication context: saml
	 *
	 * @return
	 */
	@Override
	public String getType()
	{
		return AuthType.SAML.value();
	}

	/**
	 * The SAML 2.0 Asserting Party (IDP) metadata. This can be
	 * added as a URL to the metadata file or as the XML contents
	 * of the metadata file.
	 * @return
	 */
	public String getMetadata()
	{
		return metadata;
	}

	public void setMetadata( String metadata )
	{
		initializedFields.add( "metadata" );
		this.metadata = metadata;
	}


	/**
	 * Default security groups to be associated to the administrators.
	 * Note that this property will be ignored for regular users.
	 * @return
	 */
	public List<String> getDefaultGroupIds()
	{
		return defaultGroupIds;
	}

	public void setDefaultGroupIds( List<String> defaultGroupIds )
	{
		initializedFields.add( "defaultGroupIds" );
		this.defaultGroupIds = defaultGroupIds;
	}

	/**
	 * Contains the mapping information between an external user directory and the
	 * internal Mambo users.
	 *
	 * @return
	 */
	public SamlSyncUserAttributesDto getSyncUserAttributes()
	{
		return syncUserAttributes;
	}

	public void setSyncUserAttributes( SamlSyncUserAttributesDto syncUserAttributes )
	{
		initializedFields.add( "syncUserAttributes" );
		this.syncUserAttributes = syncUserAttributes;
	}

	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
