/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.site.model.authentication;

import java.util.List;

import io.mambo.sdk.utils.ListUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates the data for the LDAP authentication method.
 */
@ToString
@EqualsAndHashCode
public class LdapAuthenticationContextDto implements AuthenticationContextDto
{
	private transient List<String> initializedFields = ListUtils.newArrayList( "type" );

	private String hostname;
	private Integer port;
	private Boolean useSsl;
	private String baseDn;
	private String userDn;
	private String password;
	private String userSearchFilter;
	private String userSearchFilterTestUser;
	private LdapSyncUserAttributesDto syncUserAttributes;

	/**
	 * The type of authentication context: ldap
	 *
	 * @return
	 */
	@Override
	public AuthType getAuthType()
	{
		return AuthType.LDAP;
	}

	/**
	 * The type of authentication context: ldap
	 *
	 * @return
	 */
	@Override
	public String getType()
	{
		return AuthType.LDAP.value();
	}

	/**
	 * Hostname of the server running LDAP. Example: ldap.domain.com
	 *
	 * @return
	 */
	public String getHostname()
	{
		return hostname;
	}

	public void setHostname( String pHostname )
	{
		initializedFields.add( "hostname" );
		hostname = pHostname;
	}

	/**
	 * The port number of the LDAP server to connect to.
	 *
	 * @return
	 */
	public Integer getPort()
	{
		return port;
	}

	public void setPort( Integer pPort )
	{
		initializedFields.add( "port" );
		port = pPort;
	}

	/**
	 * Whether we should use SSL to connect to LDAP.
	 *
	 * @return
	 */
	public Boolean getUseSsl()
	{
		return useSsl;
	}

	public void setUseSsl( Boolean pUseSsl )
	{
		initializedFields.add( "useSsl" );
		useSsl = pUseSsl;
	}

	/**
	 * Root node in LDAP from which to search for users. Example:
	 * cn=users,dc=domain,dc=com
	 *
	 * @return
	 */
	public String getBaseDn()
	{
		return baseDn;
	}

	public void setBaseDn( String pBaseDn )
	{
		initializedFields.add( "baseDn" );
		baseDn = pBaseDn;
	}

	/**
	 * The user distinguished name to use for getting
	 * authenticated contexts. Example:
	 * cn=admin,dc=example,dc=org
	 *
	 * @return
	 */
	public String getUserDn()
	{
		return userDn;
	}

	public void setUserDn( final String dn )
	{
		initializedFields.add( "userDn" );
		userDn = dn;
	}

	/**
	 * The password to connect with the LDAP authentication server.
	 *
	 * @return
	 */
	public String getPassword()
	{
		return password;
	}

	public void setPassword( final String pwd )
	{
		initializedFields.add( "password" );
		password = pwd;
	}

	/**
	 * User search filter uses RFC 2254 to query LDAP for the
	 * user which should be used during login. The filter should
	 * contain {0} which will be substituted for the username
	 * during the login process. Ensure the search filter returns
	 * a single user otherwise the login will return an error.
	 */
	public String getUserSearchFilter()
	{
		return userSearchFilter;
	}
	public void setUserSearchFilter( String userSearchFilter )
	{
		initializedFields.add( "userSearchFilter" );
		this.userSearchFilter = userSearchFilter;
	}

	/**
	 * The UUID of a user to test the userSearchFilter.
	 * Ex: ldapAdmin
	 *
	 * @return
	 */
	public String getUserSearchFilterTestUser()
	{
		return userSearchFilterTestUser;
	}

	public void setUserSearchFilterTestUser( String userSearchFilterTestUser )
	{
		initializedFields.add( "userSearchFilterTestUser" );
		this.userSearchFilterTestUser = userSearchFilterTestUser;
	}

	/**
	 * Contains the mapping information between an external user directory and the
	 * internal Mambo users.
	 *
	 * @return
	 */
	public LdapSyncUserAttributesDto getSyncUserAttributes()
	{
		return syncUserAttributes;
	}

	public void setSyncUserAttributes( LdapSyncUserAttributesDto pSyncUserAttributes )
	{
		initializedFields.add( "syncUserAttributes" );
		syncUserAttributes = pSyncUserAttributes;
	}

	@Override
	public List<String> getInitializedFields()
	{
		return initializedFields;
	}
}
