/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.site.data;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.service.site.model.authentication.AuthenticationContextDto;
import io.mambo.sdk.service.site.model.authentication.AuthenticationRedirectDto;
import io.mambo.sdk.service.site.model.authentication.HasAuthenticationContextDto;
import io.mambo.sdk.service.site.model.security.SecurityDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This object captures the data required by the Site API in
 * order to create and update sites.
 */
@ToString
@EqualsAndHashCode
public class SiteRequestData implements RequestData, HasAuthenticationContextDto
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String url;
	private String timezone;
	private Integer firstDayOfWeek;
	private SecurityDto security;
	private AuthenticationRedirectDto redirect;
	private AuthenticationContextDto authentication;


	/**
	 * The name by which this site network is known.
	 * @return name
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}


	/**
	 * The URL for this specific site network. URLs do not
	 * have to be legitimate domain names. The URLs should
	 * be easily understandable. For example:
	 * development.acme.com
	 * staging.acme.com
	 * live.acme.com
	 * @return url
	 */
	public String getUrl() { return url; }
	public void setUrl( String url ) {
		initializedFields.add( "url" );
		this.url = url;
	}


	/**
	 * The time zone to be used for this site. The time zone
	 * is used to determine when to run certain Mambo operations.
	 * The operations include:
	 * - Changing time periods in the Leaderboards
	 * - Changing time periods in KPIs
	 * - Aggregating analytics data job
	 * Time zones must be supplied using their Olson name.
	 * For example: Europe/London, Australia/Darwin, Pacific/Port_Moresby
	 * @see https://www.iana.org/time-zones
	 * @return
	 */
	public String getTimezone() { return timezone; }
	public void setTimezone( String timezone ) {
		initializedFields.add( "timezone" );
		this.timezone = timezone;
	}


	/**
	 * The first day of the week to be used for this site. For example, your week
	 * may start on Sunday or on Monday. This affects how data is stored by week.
	 * Valid range of values: 1 - 7
	 * Where 1 is Monday and 7 is Sunday.
	 * @return
	 */
	public Integer getFirstDayOfWeek() { return firstDayOfWeek; }
	public void setFirstDayOfWeek( Integer firstDayOfWeek ) {
		initializedFields.add( "firstDayOfWeek" );
		this.firstDayOfWeek = firstDayOfWeek;
	}


	/**
	 * The Security object is used to define the security settings for the site.
	 * @return
	 */
	public SecurityDto getSecurity() { return security; }
	public void setSecurity( SecurityDto security ) {
		initializedFields.add( "security" );
		this.security = security;
	}

	/**
	 * The Authentication redirect method for the Site
	 * @return
	 */
	public AuthenticationRedirectDto getRedirect(){ return redirect; }
	public void setRedirect( AuthenticationRedirectDto redirect ) {
		initializedFields.add( "redirect" );
		this.redirect = redirect;
	}

	/**
	 * The Authentication method for the site.
	 * @return
	 */
	@Override
	public AuthenticationContextDto getAuthentication(){ return authentication; }
	@Override
	public void setAuthentication( AuthenticationContextDto authentication ) {
		initializedFields.add( "authentication" );
		this.authentication = authentication;
	}

	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}