/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.site;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.site.data.SiteRequestData;
import io.mambo.sdk.service.site.model.SiteDto;
import io.mambo.sdk.service.site.param.SiteCloneParams;
import io.mambo.sdk.service.site.param.SiteDeleteParams;
import io.mambo.sdk.service.site.param.SiteGetParams;
import io.mambo.sdk.service.site.param.SiteUpdateParams;

/**
 * The SitesService class handles all Site related requests to the Mambo API.
 */
public class SitesService extends AbstractService
{
	private static final String SITES_URI = "/v1/sites";
	private static final String SITES_ID_URI = "/v1/sites/{siteId}";
	private static final String SITES_CLONE_URI = "/v1/sites/{siteId}/clone";


	public SitesService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new site.
	 *
	 * @see SiteRequestData
	 *
	 * @param data
	 *            The site request data
	 * @return
	 */
	public SiteDto create( SiteRequestData data )
	{
		return create( data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new site.
	 *
	 * @see SiteRequestData
	 *
	 * @param data
	 *            The site request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public SiteDto create( SiteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_URI )
			.responseClass( SiteDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Update an existing site by ID.
	 *
	 * @param siteId
	 *            The ID of the site to update
	 * @param data
	 *            The data with which to update the specified site object
	 * @return
	 */
	public SiteDto update( String siteId, SiteRequestData data )
	{
		return update( SiteUpdateParams.builder().siteId( siteId ).build(), data );
	}


	/**
	 * Update an existing site by ID.
	 *
	 * @param params
	 *            The parameters required to update the site
	 * @param data
	 *            The data with which to update the specified site object
	 * @return
	 */
	public SiteDto update( SiteUpdateParams params, SiteRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing site by ID.
	 *
	 * @param params
	 *            The parameters required to update the site
	 * @param data
	 *            The data with which to update the specified site object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public SiteDto update( SiteUpdateParams params, SiteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_ID_URI )
			.responseClass( SiteDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a site.
	 *
	 * @param siteId
	 *            The ID of the site to clone
	 * @param data
	 *            The data of the site into which we wish to clone
	 * @return
	 */
	public SiteDto clone( String siteId, SiteRequestData data )
	{
		return clone( SiteCloneParams.builder().siteId( siteId ).build(), data );
	}


	/**
	 * Clone a site.
	 *
	 * @param params
	 *            The parameters required to clone the site
	 * @param data
	 *            The data of the site into which we wish to clone
	 * @return
	 */
	public SiteDto clone( SiteCloneParams params, SiteRequestData data )
	{
		return clone( params, data, RequestOptions.create() );
	}


	/**
	 * Clone a site.
	 *
	 * @param params
	 *            The parameters required to clone the site
	 * @param data
	 *            The data of the site into which we wish to clone
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public SiteDto clone( SiteCloneParams params, SiteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_CLONE_URI )
			.responseClass( SiteDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a site by ID.
	 *
	 * @param siteId
	 *            The ID of the site to delete
	 * @return
	 */
	public Status delete( String siteId )
	{
		return delete( SiteDeleteParams.builder().siteId( siteId ).build() );
	}


	/**
	 * Delete a site by ID.
	 *
	 * @param params
	 *            The parameters required to delete the site
	 * @return
	 */
	public Status delete( SiteDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a site by ID.
	 *
	 * @param params
	 *            The parameters required to delete the site
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( SiteDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a site by ID
	 *
	 * @param siteId
	 *            The ID of the site to retrieve
	 * @return
	 */
	public SiteDto get( String siteId )
	{
		return get( SiteGetParams.builder().siteId( siteId ).build() );
	}


	/**
	 * Get a site by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the site
	 * @return
	 */
	public SiteDto get( SiteGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a site by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the site
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public SiteDto get( SiteGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_ID_URI )
			.responseClass( SiteDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of sites by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deleteSites( DeleteRequestData data )
	{
		return deleteSites( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of sites by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteSites( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * Get the list of sites
	 *
	 * @return
	 */
	public List<SiteDto> getSites()
	{
		return getSites( RequestOptions.create() );
	}


	/**
	 * Get the list of sites
	 *
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<SiteDto> getSites( RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( SITES_URI )
			.responseClass( SiteDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Return an empty {@link SiteRequestData} object
	 *
	 * @return
	 */
	public SiteRequestData newSiteRequestData()
	{
		return new SiteRequestData();
	}
}