/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.reward.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.PrizeDto;
import io.mambo.sdk.service.common.model.ScarcityDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.common.model.tag.AbstractHasAwardByTagRequestData;
import io.mambo.sdk.service.reward.data.RewardRequestData;
import io.mambo.sdk.service.reward.model.attributes.RewardAttrs;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Reward data shared by {@link RewardDto} and {@link RewardRequestData}
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public abstract class AbstractReward extends AbstractHasAwardByTagRequestData
		implements HasTranslation, HasCustomFields
{
	public enum RewardType {
		LEVEL("level"), ACHIEVEMENT("achievement"), MISSION("mission"), GIFT("gift");
		private String value;
		private RewardType( String value ) { this.value = value; }
		public String value() { return value; }
		public Boolean isValue( String other ) { return StringUtils.equals( value, other ); }
	}

	private String name;
	private String message;
	private String hint;
	private String startDate;
	private String endDate;
	private Boolean active;
	private Boolean hideInWidgets;
	private Boolean multiUnlock;
	private Long hasAtLeast;
	private Long countLimit;
	private RewardAttrs attrs;
	private PrizeDto prizes;
	private ScarcityDto scarcity;
	private List<UnlockConditionDto> unlockConditions;
	private List<CustomFieldValueDto> customFields;
	private List<RewardTranslationDto> translations;


	/**
	 * The name of the reward. See the achievement or level pages in
	 * administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * Indicates whether the reward is active or not.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * The message associated with the reward. See the achievement or
	 * level pages in administration panel for more information.
	 * @return
	 */
	public String getMessage() { return message; }
	public void setMessage( String message ) {
		initializedFields.add( "message" );
		this.message = message;
	}

	/**
	 * The hint associated with the reward. See the achievement or level
	 * pages in administration panel for more information.
	 * @return
	 */
	public String getHint() { return hint; }
	public void setHint( String hint ) {
		initializedFields.add( "hint" );
		this.hint = hint;
	}

	/**
	 * Indicates whether the reward's hint should be hidden or not. See the
	 * achievement or level pages in administration panel for more information.
	 * @return
	 */
	public Boolean getHideInWidgets() { return hideInWidgets; }
	public void setHideInWidgets( Boolean hideInWidgets ) {
		initializedFields.add( "hideInWidgets" );
		this.hideInWidgets = hideInWidgets;
	}

	/**
	 * The prizes object contains the prizes that a user will earn
	 * when unlocking this reward.
	 * @return
	 */
	public PrizeDto getPrizes() { return prizes; }
	public void setPrizes( PrizeDto prizes ) {
		initializedFields.add( "prizes" );
		this.prizes = prizes;
	}

	/**
	 * Determines whether the reward can be unlocked multiple times. This
	 * means that users can unlock this reward every time they complete the
	 * criteria associated to the reward. For example, you can create an
	 * achievement for completing 5 training courses, with multi unlock the
	 * users will earn the achievement every time they complete 5 courses.
	 * Note that this property will take precedence over any expiration
	 * defined in the achievements.
	 * @return
	 */
	public Boolean getMultiUnlock() { return multiUnlock; }
	public void setMultiUnlock( Boolean multiUnlock ) {
		initializedFields.add( "multiUnlock" );
		this.multiUnlock = multiUnlock;
	}

	/**
	 * Determines whether the reward should be limited to a specific number of
	 * unlocks per user. This is useful in combination with multi unlock rewards
	 * as well as with expiring achievements.
	 * @return
	 */
	public Long getCountLimit() { return countLimit; }
	public void setCountLimit( Long countLimit ) {
		initializedFields.add( "countLimit" );
		this.countLimit = countLimit;
	}

	/**
	 * The scarcity represents the overall number of times that a particular
	 * reward can be unlocked. This effectively creates a limited series
	 * reward which can only be unlocked X times.
	 * @return
	 */
	public ScarcityDto getScarcity() { return scarcity; }
	public void setScarcity( ScarcityDto scarcity ) {
		initializedFields.add( "scarcity" );
		this.scarcity = scarcity;
	}

	/**
	 * Determines how many of the reward's conditions need to be met in order
	 * for the reward to be unlocked. Set to zero if you require the user to
	 * complete all the conditions.
	 * For example: if you have an achievement composed of 5 different behaviour
	 * repetitions and you only require 3 of those to be completed, you would
	 * set hasAtLeast equal to 3.
	 * @return
	 */
	public Long getHasAtLeast() { return hasAtLeast; }
	public void setHasAtLeast( Long hasAtLeast ) {
		initializedFields.add( "hasAtLeast" );
		this.hasAtLeast = hasAtLeast;
	}

	/**
	 * This represents the date from which this reward can be unlocked by users.
	 * If no date is specified, the reward can always be unlocked.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getStartDate() { return startDate; }
	public void setStartDate( String startDate ) {
		initializedFields.add( "startDate" );
		this.startDate = startDate;
	}

	/**
	 * This represents the date from which this reward can no longer be unlocked by users
	 * If no date is specified, the reward can always be unlocked.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getEndDate() { return endDate; }
	public void setEndDate( String endDate ) {
		initializedFields.add( "endDate" );
		this.endDate = endDate;
	}

	/**
	 * The attributes of the reward. There are currently two types of
	 * attributes: AchievementAttrs and LevelAttrs.
	 * @return
	 */
	public RewardAttrs getAttrs() { return attrs; }
	public void setAttrs( RewardAttrs attrs ) {
		initializedFields.add( "attrs" );
		this.attrs = attrs;
	}

	/**
	 * True if the reward is a level
	 * @return
	 */
	public Boolean isLevel() {
		if( attrs == null ) {
			return false;
		}
		return RewardType.LEVEL.isValue( attrs.getType() );
	}

	/**
	 * True if the reward is an achievement
	 * @return
	 */
	public Boolean isAchievement() {
		if( attrs == null ) {
			return false;
		}
		return RewardType.ACHIEVEMENT.isValue( attrs.getType() );
	}

	/**
	 * True if the reward is a mission
	 * @return
	 */
	public Boolean isMission() {
		if( attrs == null ) {
			return false;
		}
		return RewardType.MISSION.isValue( attrs.getType() );
	}

	/**
	 * True if the reward is a gift
	 * @return
	 */
	public Boolean isGift() {
		if( attrs == null ) {
			return false;
		}
		return RewardType.GIFT.isValue( attrs.getType() );
	}

	/**
	 * The list of unlock conditions required to earn this reward
	 * @return
	 */
	public List<UnlockConditionDto> getUnlockConditions() { return unlockConditions; }
	public void setUnlockConditions( List<UnlockConditionDto> unlockConditions ) {
		initializedFields.add( "unlockConditions" );
		this.unlockConditions = unlockConditions;
	}
	public void addUnlockCondition( UnlockConditionDto unlockCondition ) {
		initializedFields.add( "unlockConditions" );
		if( unlockConditions == null ) {
			unlockConditions = new ArrayList<>();
		}
		unlockConditions.add( unlockCondition );
	}

	/**
	 * Custom fields defined for the reward. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the reward.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<RewardTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<RewardTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( RewardTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public RewardTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( RewardTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}
}
