/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.reward;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.reward.data.RewardRequestData;
import io.mambo.sdk.service.reward.model.RewardDto;
import io.mambo.sdk.service.reward.param.RewardCloneParams;
import io.mambo.sdk.service.reward.param.RewardCreateParams;
import io.mambo.sdk.service.reward.param.RewardCustomFieldParams;
import io.mambo.sdk.service.reward.param.RewardDeleteParams;
import io.mambo.sdk.service.reward.param.RewardGetAchievementListParams;
import io.mambo.sdk.service.reward.param.RewardGetGiftListParams;
import io.mambo.sdk.service.reward.param.RewardGetLevelListParams;
import io.mambo.sdk.service.reward.param.RewardGetListParams;
import io.mambo.sdk.service.reward.param.RewardGetMissionListParams;
import io.mambo.sdk.service.reward.param.RewardGetParams;
import io.mambo.sdk.service.reward.param.RewardUpdateParams;
import io.mambo.sdk.service.reward.param.RewardUploadParams;

/**
 * The RewardsService class handles all Reward related requests to the Mambo
 * API.
 */
public class RewardsService extends AbstractService
{
	private static final String REWARDS_URI = "/v1/rewards";
	private static final String REWARDS_ID_URI = REWARDS_URI + "/{rewardId}";
	private static final String REWARDS_IMAGE_URI = REWARDS_ID_URI + "/image";
	private static final String REWARDS_CLONE_URI = REWARDS_ID_URI + "/clone";
	private static final String REWARDS_CUSTOM_URI = REWARDS_ID_URI + "/custom_fields";
	private static final String REWARDS_CUSTOM_IMAGE_URI = REWARDS_ID_URI + "/custom_fields/image";

	private static final String REWARDS_SITE_URI = "/v1/{siteUrl}/rewards";
	private static final String ACHIEVEMENTS_SITE_URI = REWARDS_SITE_URI + "/achievements";
	private static final String LEVELS_SITE_URI = REWARDS_SITE_URI + "/levels";
	private static final String MISSIONS_SITE_URI = REWARDS_SITE_URI + "/missions";
	private static final String GIFTS_SITE_URI = REWARDS_SITE_URI + "/gifts";


	public RewardsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new reward.
	 *
	 * @see RewardRequestData
	 *
	 * @param siteUrl
	 *            The site to which the reward belongs to
	 * @param data
	 *            The reward request data
	 * @return
	 */
	public RewardDto create( String siteUrl, RewardRequestData data )
	{
		return create( RewardCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * This method is used to create a new reward.
	 *
	 * @see RewardRequestData
	 *
	 * @param siteUrl
	 *            The site to which the reward belongs to
	 * @param data
	 *            The reward request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto create( String siteUrl, RewardRequestData data, RequestOptions requestOptions )
	{
		RewardCreateParams params = RewardCreateParams.builder().siteUrl( siteUrl ).build();
		return create( params, data, requestOptions );
	}


	/**
	 * This method is used to create a new reward.
	 *
	 * @param params
	 *            The parameters required to create the reward
	 * @param data
	 *            The reward request data
	 * @return
	 */
	public RewardDto create( RewardCreateParams params, RewardRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new reward.
	 *
	 * @param params
	 *            The parameters required to create the reward
	 * @param data
	 *            The reward request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto create(
		RewardCreateParams params,
		RewardRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_SITE_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing reward by ID.
	 *
	 * @param rewardId
	 *            The ID of the reward to update
	 * @param data
	 *            The data with which to update the specified reward object
	 * @return
	 */
	public RewardDto update( String rewardId, RewardRequestData data )
	{
		return update( RewardUpdateParams.builder()
			.rewardId( rewardId )
			.build(), data );
	}


	/**
	 * Update an existing reward by ID.
	 *
	 * @param params
	 *            The parameters required to update the reward
	 * @param data
	 *            The data with which to update the specified reward object
	 * @return
	 */
	public RewardDto update( RewardUpdateParams params, RewardRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing reward by ID.
	 *
	 * @param params
	 *            The parameters required to update the reward
	 * @param data
	 *            The data with which to update the specified reward object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto update(
		RewardUpdateParams params,
		RewardRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_ID_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Upload an image for the reward
	 *
	 * @param rewardId
	 *            The ID of the reward for which to upload the image
	 * @param image
	 *            The image to upload for the reward
	 * @return
	 */
	public RewardDto uploadImage( String rewardId, File image )
	{
		return uploadImage( RewardUploadParams.builder()
			.rewardId( rewardId )
			.build(), image );
	}


	/**
	 * Upload an image for the reward
	 *
	 * @param params
	 *            The parameters required to upload an image for the reward
	 * @param image
	 *            The image to upload for the reward
	 * @return
	 */
	public RewardDto uploadImage( RewardUploadParams params, File image )
	{
		return uploadImage( params, image, RequestOptions.create() );
	}


	/**
	 * Upload an image for the reward
	 *
	 * @param params
	 *            The parameters required to upload an image for the reward
	 * @param image
	 *            The image to upload for the reward
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto uploadImage(
		RewardUploadParams params,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_IMAGE_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image ) )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a reward
	 *
	 * @param rewardId
	 *            The ID of the reward to clone
	 * @return
	 */
	public RewardDto clone( String rewardId )
	{
		return clone( RewardCloneParams.builder()
			.rewardId( rewardId )
			.build() );
	}


	/**
	 * Clone a reward
	 *
	 * @param params
	 *            The parameters required to clone the reward
	 * @return
	 */
	public RewardDto clone( RewardCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a reward
	 *
	 * @param params
	 *            The parameters required to clone the reward
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto clone( RewardCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_CLONE_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a reward by ID.
	 *
	 * @param rewardId
	 *            The ID of the reward to delete
	 * @return
	 */
	public Status delete( String rewardId )
	{
		return delete( RewardDeleteParams.builder()
			.rewardId( rewardId )
			.build() );
	}


	/**
	 * Delete a reward by ID.
	 *
	 * @param params
	 *            The parameters required to delete the reward
	 * @return
	 */
	public Status delete( RewardDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a reward by ID.
	 *
	 * @param params
	 *            The parameters required to delete the reward
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( RewardDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a reward by ID
	 *
	 * @param rewardId
	 *            The ID of the reward to retrieve
	 * @return
	 */
	public RewardDto get( String rewardId )
	{
		return get( RewardGetParams.builder()
			.rewardId( rewardId )
			.build() );
	}


	/**
	 * Get a reward by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the reward
	 * @return
	 */
	public RewardDto get( RewardGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a reward by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the reward
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto get( RewardGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_ID_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of rewards by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deleteRewards( DeleteRequestData data )
	{
		return deleteRewards( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of rewards by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteRewards( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get the list of rewards for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the rewards
	 * @return
	 */
	public List<RewardDto> getRewards( String siteUrl )
	{
		return getRewards( RewardGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of rewards for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of rewards
	 * @return
	 */
	public List<RewardDto> getRewards( RewardGetListParams params )
	{
		return getRewards( params, RequestOptions.create() );
	}


	/**
	 * Get a list of rewards for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of rewards
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<RewardDto> getRewards( RewardGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_SITE_URI )
			.responseClass( RewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of achievements for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the rewards
	 * @return
	 */
	public List<RewardDto> getAchievements( String siteUrl )
	{
		return getAchievements( RewardGetAchievementListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of achievements for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of achievements
	 * @return
	 */
	public List<RewardDto> getAchievements( RewardGetAchievementListParams params )
	{
		return getAchievements( params, RequestOptions.create() );
	}


	/**
	 * Get a list of achievements for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of achievements
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<RewardDto> getAchievements(
		RewardGetAchievementListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( ACHIEVEMENTS_SITE_URI )
			.responseClass( RewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of levels for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the rewards
	 * @return
	 */
	public List<RewardDto> getLevels( String siteUrl )
	{
		return getLevels( RewardGetLevelListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of levels for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of levels
	 * @return
	 */
	public List<RewardDto> getLevels( RewardGetLevelListParams params )
	{
		return getLevels( params, RequestOptions.create() );
	}


	/**
	 * Get a list of levels for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of levels
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<RewardDto> getLevels(
		RewardGetLevelListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEVELS_SITE_URI )
			.responseClass( RewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of missions for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the rewards
	 * @return
	 */
	public List<RewardDto> getMissions( String siteUrl )
	{
		return getMissions( RewardGetMissionListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of missions for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of missions
	 * @return
	 */
	public List<RewardDto> getMissions( RewardGetMissionListParams params )
	{
		return getMissions( params, RequestOptions.create() );
	}


	/**
	 * Get a list of missions for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of missions
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<RewardDto> getMissions(
		RewardGetMissionListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( MISSIONS_SITE_URI )
			.responseClass( RewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of gifts for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the rewards
	 * @return
	 */
	public List<RewardDto> getGifts( String siteUrl )
	{
		return getGifts( RewardGetGiftListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of gifts for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of gifts
	 * @return
	 */
	public List<RewardDto> getGifts( RewardGetGiftListParams params )
	{
		return getGifts( params, RequestOptions.create() );
	}


	/**
	 * Get a list of gifts for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve a list of gifts
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<RewardDto> getGifts(
		RewardGetGiftListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( GIFTS_SITE_URI )
			.responseClass( RewardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * This method is used to add custom fields to an existing reward.
	 *
	 * @param rewardId
	 *            The reward to which to add the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto addCustomFields( String rewardId, CustomFieldValueRequestData data )
	{
		RewardCustomFieldParams params = RewardCustomFieldParams.builder().rewardId( rewardId ).build();
		return addCustomFields( params, data );
	}


	/**
	 * This method is used to add custom fields to an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto addCustomFields( RewardCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return addCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to add custom fields to an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto addCustomFields(
		RewardCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.POST, params, data, requestOptions );
	}


	/**
	 * This method is used to update custom fields of an existing reward.
	 *
	 * @param rewardId
	 *            The reward to update the custom fields of
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto updateCustomFields( String rewardId, CustomFieldValueRequestData data )
	{
		RewardCustomFieldParams params = RewardCustomFieldParams.builder().rewardId( rewardId ).build();
		return updateCustomFields( params, data );
	}


	/**
	 * This method is used to update custom fields of an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto updateCustomFields( RewardCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return updateCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to update custom fields of an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto updateCustomFields(
		RewardCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.PUT, params, data, requestOptions );
	}


	/**
	 * This method is used to delete custom fields from an existing reward.
	 *
	 * @param rewardId
	 *            The reward to delete the custom fields from
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto deleteCustomFields( String rewardId, CustomFieldValueRequestData data )
	{
		RewardCustomFieldParams params = RewardCustomFieldParams.builder().rewardId( rewardId ).build();
		return deleteCustomFields( params, data );
	}


	/**
	 * This method is used to delete custom fields from an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public RewardDto deleteCustomFields( RewardCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return deleteCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to delete custom fields from an existing reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto deleteCustomFields(
		RewardCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, requestOptions );
	}


	/**
	 * Generic method used to manipulate the custom fields
	 *
	 * @param method
	 *            The HTTP method to use (POST, PUT, DELETE)
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	private RewardDto modCustomFields(
		HttpMethod method,
		RewardCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_CUSTOM_URI )
			.responseClass( RewardDto.class )
			.method( method )
			.params( params )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing
	 * reward.
	 *
	 * @param rewardId
	 *            The ID of the reward for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the reward
	 * @return
	 */
	public RewardDto addCustomField(
		String rewardId,
		CustomFieldValueDto data,
		File image )
	{
		RewardCustomFieldParams params = RewardCustomFieldParams.builder().rewardId( rewardId ).build();
		return addCustomField( params, data, image );
	}


	/**
	 * This method is used to add a custom field with an image to an existing
	 * reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the reward
	 * @return
	 */
	public RewardDto addCustomField(
		RewardCustomFieldParams params,
		CustomFieldValueDto data,
		File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing
	 * reward.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the reward
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public RewardDto addCustomField(
		RewardCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( REWARDS_CUSTOM_IMAGE_URI )
			.responseClass( RewardDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.requestData( new FileRequestData( image, data ) )
			.build() );
	}


	/**
	 * Return an empty {@link RewardRequestData} object
	 *
	 * @return
	 */
	public RewardRequestData newRewardRequestData()
	{
		return new RewardRequestData();
	}
}