/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.point.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class PointTransactionGetByActivityIdParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();

	private PointTransactionGetByActivityIdParams(
		Builder builder )
	{
		params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}

	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		private static final String ACTIVITY_ID_MUST_NOT_BE_NULL = "activityId must not be null";

		/**
		 * The ID of the activity with which to retrieve the transactions
		 */
		public Builder activityId( String activityId )
		{
			checkNotNull( activityId, ACTIVITY_ID_MUST_NOT_BE_NULL );
			params.put( "activityId", activityId );
			return this;
		}


		/**
		 * Filter transactions by point ID
		 */
		public Builder pointId( String pointId )
		{
			params.put( "pointId", pointId );
			return this;
		}


		/**
		 * Filter transactions by user ID
		 */
		public Builder userId( String userId )
		{
			params.put( "userId", userId );
			return this;
		}


		/**
		 * Filter transactions by transaction type
		 */
		public Builder transactionType( String transactionType )
		{
			params.put( "transactionType", transactionType );
			return this;
		}


		/**
		 * Filter transactions by status
		 */
		public Builder status( String status )
		{
			params.put( "status", status );
			return this;
		}


		/**
		 * Include detailed point information in the response
		 */
		public Builder withPoints( boolean withPoints )
		{
			params.put( "withPoints", String.valueOf( withPoints ) );
			return this;
		}


		/**
		 * Include detailed user information in the response
		 */
		public Builder withUsers( boolean withUsers )
		{
			params.put( "withUsers", String.valueOf( withUsers ) );
			return this;
		}


		public PointTransactionGetByActivityIdParams build()
		{
			checkContainsKey( params, "activityId", ACTIVITY_ID_MUST_NOT_BE_NULL );
			return new PointTransactionGetByActivityIdParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}