/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.point.model;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.utils.NumberConverter;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * The point transaction contains information about events
 * that caused changes to the user's point wallet.
 */
@ToString
@EqualsAndHashCode
public class PointTransactionDto
{
	private String id;
	private String transactionType;
	private Number points;
	private Number redeemablePoints;
	private String expiryOn;
	private boolean canBeNegative;
	private boolean debited;
	private boolean rejected;
	private TransactionSourceActivityDto source;
	private TransactionParentDto parent;
	private String processedOn;


	/**
	 * The ID that uniquely identifies this transaction
	 * @return
	 */
	public String getId() { return id; }
	public void setId( String id ) { this.id = id; }


	/**
	 * The type of the transaction.
	 * Available types: increment, redeem, refund, expired, set
	 * @return
	 */
	public String getTransactionType() { return transactionType; }
	public void setTransactionType( String transactionType ) { this.transactionType = transactionType; }


	/**
	 * The number of points applied to the user as part of the transaction
	 * @return
	 */
	public Number getPoints() { return points; }
	public void setPoints( Number points ) { this.points = points; }

	@JsonIgnore
	public NumberConverter getPointsHelper() { return new NumberConverter( points ); }


	/**
	 * The number of points available to spend from this transaction
	 * @return
	 */
	public Number getRedeemablePoints() { return redeemablePoints; }
	public void setRedeemablePoints( Number redeemablePoints ) { this.redeemablePoints = redeemablePoints; }


	/**
	 * The date on which the point is going to expire or has expired on (if applicable).
	 * This will be a UTC timestamp in ISO 8601 format with millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * For example: 2013-01-20T20:43:24.094Z
	 * @return
	 */
	public String getExpiryOn() { return expiryOn; }
	public void setExpiryOn( String expiryOn ) { this.expiryOn = expiryOn; }


	/**
	 * Indicates whether at the time of this transaction the user's points were allowed to be negative.
	 * @return
	 */
	public boolean getCanBeNegative() { return canBeNegative; }
	public void setCanBeNegative( boolean canBeNegative ) { this.canBeNegative = canBeNegative; }


	/**
	 * True if the point is expired and the points have been removed from the relevant user's wallet.
	 * @return
	 */
	public boolean getDebited() { return debited; }
	public void setDebited( boolean debited ) { this.debited = debited; }


	/**
	 * True if the point is rejected.
	 * @return
	 */
	public boolean getRejected() { return rejected; }
	public void setRejected( boolean rejected ) { this.rejected = rejected; }


	/**
	 * The source activity that generated this transaction
	 * @return
	 */
	public TransactionSourceActivityDto getSource() { return source; }
	public void setSource( TransactionSourceActivityDto source ) { this.source = source; }


	/**
	 * The parent mechanic which awarded the points of the transaction
	 * @return
	 */
	public TransactionParentDto getParent() { return parent; }
	public void setParent( TransactionParentDto parent ) { this.parent = parent; }


	/**
	 * The date on which the transaction was processed.
	 * @return
	 */
	public String getProcessedOn() { return processedOn; }
	public void setProcessedOn( String processedOn ) { this.processedOn = processedOn; }
}
