/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.point.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasRecurrence;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.common.model.recurrence.FixedPeriodRecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.NeverRecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.RecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.VariablePeriodRecurrenceDto;
import io.mambo.sdk.service.point.data.PointRequestData;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Point data shared by {@link PointDto} and {@link PointRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractPoint implements HasTranslation, HasRecurrence, HasCustomFields
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String description;
	private Boolean active;
	private Boolean hideInSummary;
	private Boolean stopAtZero;
	private Boolean negativeable;
	private Boolean redeemable;
	private Boolean giftable;
	private Boolean bountiable;
	private Boolean expirable;
	private Boolean settable;
	private RecurrenceDto expiration;
	private DecimalsDto decimals;
	private UnitsDto units;
	private List<CustomFieldValueDto> customFields;
	private List<PointTranslationDto> translations;

	/**
	 * The name of the point.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * Indicates whether the point is active or not.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * Indicates whether the point should be hidden in the summary
	 * of the Profile and Header widgets. This is usually associated
	 * to KPIs or points which are used only for tracking.
	 * @return
	 */
	public Boolean getHideInSummary() { return hideInSummary; }
	public void setHideInSummary( Boolean hideInSummary ) {
		initializedFields.add( "hideInSummary" );
		this.hideInSummary = hideInSummary;
	}

	/**
	 * Indicates whether we should stop at zero when we are subtracting these
	 * points from a user. This will skip the user's balance validation as the
	 * points will never go below zero so the user is considered to always have
	 * sufficient balance.
	 * For example: the user has 5 points, we subtract 10 points, the user now
	 * has 0 points (instead of -5).
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getStopAtZero() { return stopAtZero; }
	public void setStopAtZero( Boolean stopAtZero ) {
		initializedFields.add( "stopAtZero" );
		this.stopAtZero = stopAtZero;
	}


	/**
	 * Indicates whether this point can be set.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getSettable() { return settable; }
	public void setSettable( Boolean settable ) {
		initializedFields.add( "settable" );
		this.settable = settable;
	}

	/**
	 * Indicates whether the user can have negative points of this type.
	 * If the points cannot be negative then no user will ever have less
	 * than 0 points of this type.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getNegativeable() { return negativeable; }
	public void setNegativeable( Boolean negativeable ) {
		initializedFields.add( "negativeable" );
		this.negativeable = negativeable;
	}

	/**
	 * Indicates whether the points can be redeemed in exchange for items.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getRedeemable() { return redeemable; }
	public void setRedeemable( Boolean redeemable ) {
		initializedFields.add( "redeemable" );
		this.redeemable = redeemable;
	}

	/**
	 * Indicates whether the points can be gifted to other users.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getGiftable() { return giftable; }
	public void setGiftable( Boolean giftable ) {
		initializedFields.add( "giftable" );
		this.giftable = giftable;
	}

	/**
	 * Indicates whether the points can be used as a bounty which will then
	 * be given to the user who meets the bounty's requirements.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getBountiable() { return bountiable; }
	public void setBountiable( Boolean bountiable ) {
		initializedFields.add( "bountiable" );
		this.bountiable = bountiable;
	}

	/**
	 * Indicates whether the points expire.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public Boolean getExpirable() { return expirable; }
	public void setExpirable( Boolean expirable ) {
		initializedFields.add( "expirable" );
		this.expirable = expirable;
	}

	/**
	 * If the points are expirable then this field should contain the general
	 * expiration information.
	 * See the {@link NeverRecurrenceDto}, {@link FixedPeriodRecurrenceDto} and
	 * {@link VariablePeriodRecurrenceDto} objects for more information.
	 */
	@Override
	public RecurrenceDto getExpiration() { return expiration; }
	@Override
	public void setExpiration( RecurrenceDto expiration ) {
		initializedFields.add( "expiration" );
		this.expiration = expiration;
	}

	/**
	 * The description associated with the point.
	 * See the points page in administration panel for more information.
	 * @return
	 */
	public String getDescription() { return description; }
	public void setDescription( String description ) {
		initializedFields.add( "description" );
		this.description = description;
	}

	/**
	 * This contains the point's decimal places information.
	 * Note that this value can only be set when the point is being
	 * created so choose wisely.
	 * See the {@link DecimalsDto} object for more information.
	 */
	public DecimalsDto getDecimals() { return decimals; }
	public void setDecimals( DecimalsDto decimals ) {
		initializedFields.add( "decimals" );
		this.decimals = decimals;
	}

	/**
	 * This contains the point's units of measure.
	 * See the {@link UnitsDto} object for more information.
	 */
	public UnitsDto getUnits() { return units; }
	public void setUnits( UnitsDto units ) {
		initializedFields.add( "units" );
		this.units = units;
	}

	/**
	 * Custom fields defined for the point. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the point.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<PointTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<PointTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( PointTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public PointTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( PointTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}
}
