/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.point;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.ResponseEntity;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.point.data.PointRequestData;
import io.mambo.sdk.service.point.model.PointDto;
import io.mambo.sdk.service.point.model.PointTransactionDto;
import io.mambo.sdk.service.point.model.PointWalletDto;
import io.mambo.sdk.service.point.param.PointCloneParams;
import io.mambo.sdk.service.point.param.PointCreateParams;
import io.mambo.sdk.service.point.param.PointCustomFieldParams;
import io.mambo.sdk.service.point.param.PointDeleteParams;
import io.mambo.sdk.service.point.param.PointGetListParams;
import io.mambo.sdk.service.point.param.PointGetParams;
import io.mambo.sdk.service.point.param.PointTransactionGetByActivityIdParams;
import io.mambo.sdk.service.point.param.PointUpdateParams;
import io.mambo.sdk.service.point.param.PointUploadParams;
import io.mambo.sdk.service.point.param.PointWalletsGetListParams;

/**
 * The PointsService class handles all Point related requests to the Mambo API.
 */
public class PointsService extends AbstractService
{
	private static final String POINTS_URI = "/v1/points";
	private static final String POINTS_ID_URI = POINTS_URI + "/{pointId}";
	private static final String POINTS_IMAGE_URI = POINTS_ID_URI + "/image";
	private static final String POINTS_CLONE_URI = POINTS_ID_URI + "/clone";
	private static final String POINTS_CUSTOM_URI = POINTS_ID_URI + "/custom_fields";
	private static final String POINTS_CUSTOM_IMAGE_URI = POINTS_ID_URI + "/custom_fields/image";
	private static final String POINTS_TRANSACTIONS_BY_ACTIVITY_URI = POINTS_ID_URI + "/transactions/{activityId}";
	private static final String POINTS_SITE_URI = "/v1/{siteUrl}/points";
	private static final String POINTS_SITE_WALLETS_URI = "/v1/{siteUrl}/points/wallets";


	public PointsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new point.
	 *
	 * @see PointRequestData
	 *
	 * @param siteUrl
	 *            The site to which the point belongs to
	 * @param data
	 *            The point request data
	 * @return
	 */
	public PointDto create( String siteUrl, PointRequestData data )
	{
		return create( siteUrl, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new point.
	 *
	 * @see PointRequestData
	 *
	 * @param siteUrl
	 *            The site to which the point belongs to
	 * @param data
	 *            The point request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto create( String siteUrl, PointRequestData data, RequestOptions requestOptions )
	{
		return create( PointCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new point.
	 *
	 * @param params
	 *            The parameters required to create the point
	 * @param data
	 *            The point request data
	 * @return
	 */
	public PointDto create( PointCreateParams params, PointRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new point.
	 *
	 * @param params
	 *            The parameters required to create the point
	 * @param data
	 *            The point request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto create(
		PointCreateParams params,
		PointRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_SITE_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing point by ID.
	 *
	 * @param pointId
	 *            The ID of the point to update
	 * @param data
	 *            The data with which to update the specified point object
	 * @return
	 */
	public PointDto update( String pointId, PointRequestData data )
	{
		return update( PointUpdateParams.builder()
			.pointId( pointId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Update an existing point by ID.
	 *
	 * @param pointId
	 *            The ID of the point to update
	 * @param data
	 *            The data with which to update the specified point object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto update( String pointId, PointRequestData data, RequestOptions requestOptions )
	{
		return update( PointUpdateParams.builder()
			.pointId( pointId )
			.build(), data, requestOptions );
	}


	/**
	 * Update an existing point by ID.
	 *
	 * @param params
	 *            The parameters required to update the point
	 * @param data
	 *            The data with which to update the specified point object
	 * @return
	 */
	public PointDto update( PointUpdateParams params, PointRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing point by ID.
	 *
	 * @param params
	 *            The parameters required to update the point
	 * @param data
	 *            The data with which to update the specified point object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto update(
		PointUpdateParams params,
		PointRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_ID_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Upload an image for the point
	 *
	 * @param pointId
	 *            The ID of the point for which to upload the image
	 * @param image
	 *            The image to upload for the point
	 * @return
	 */
	public PointDto uploadImage( String pointId, File image )
	{
		return uploadImage( PointUploadParams.builder()
			.pointId( pointId )
			.build(), image, RequestOptions.create() );
	}


	/**
	 * Upload an image for the point
	 *
	 * @param pointId
	 *            The ID of the point for which to upload the image
	 * @param image
	 *            The image to upload for the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto uploadImage( String pointId, File image, RequestOptions requestOptions )
	{
		return uploadImage( PointUploadParams.builder()
			.pointId( pointId )
			.build(), image, requestOptions );
	}


	/**
	 * Upload an image for the point
	 *
	 * @param params
	 *            The parameters required to upload an image for the point
	 * @param image
	 *            The image to upload for the point
	 * @return
	 */
	public PointDto uploadImage( PointUploadParams params, File image )
	{
		return uploadImage( params, image, RequestOptions.create() );
	}


	/**
	 * Upload an image for the point
	 *
	 * @param params
	 *            The parameters required to upload an image for the point
	 * @param image
	 *            The image to upload for the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto uploadImage( PointUploadParams params, File image, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_IMAGE_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image ) )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a point
	 *
	 * @param pointId
	 *            The ID of the point to clone
	 * @return
	 */
	public PointDto clone( String pointId )
	{
		return clone( PointCloneParams.builder()
			.pointId( pointId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Clone a point
	 *
	 * @param pointId
	 *            The ID of the point to clone
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto clone( String pointId, RequestOptions requestOptions )
	{
		return clone( PointCloneParams.builder()
			.pointId( pointId )
			.build(), requestOptions );
	}


	/**
	 * Clone a point
	 *
	 * @param params
	 *            The parameters required to clone the point
	 * @return
	 */
	public PointDto clone( PointCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a point
	 *
	 * @param params
	 *            The parameters required to clone the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto clone( PointCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_CLONE_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a point by ID.
	 *
	 * @param pointId
	 *            The ID of the point to delete
	 * @return
	 */
	public Status delete( String pointId )
	{
		return delete( PointDeleteParams.builder()
			.pointId( pointId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Delete a point by ID.
	 *
	 * @param pointId
	 *            The ID of the point to delete
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( String pointId, RequestOptions requestOptions )
	{
		return delete( PointDeleteParams.builder()
			.pointId( pointId )
			.build(), requestOptions );
	}


	/**
	 * Delete a point by ID.
	 *
	 * @param params
	 *            The parameters required to delete the point
	 * @return
	 */
	public Status delete( PointDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a point by ID.
	 *
	 * @param params
	 *            The parameters required to delete the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( PointDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a point by ID
	 *
	 * @param pointId
	 *            The ID of the point to retrieve
	 * @return
	 */
	public PointDto get( String pointId )
	{
		return get( PointGetParams.builder()
			.pointId( pointId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a point by ID
	 *
	 * @param pointId
	 *            The ID of the point to retrieve
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto get( String pointId, RequestOptions requestOptions )
	{
		return get( PointGetParams.builder()
			.pointId( pointId )
			.build(), requestOptions );
	}


	/**
	 * Get a point by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the point
	 * @return
	 */
	public PointDto get( PointGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a point by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto get( PointGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_ID_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of points by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deletePoints( DeleteRequestData data )
	{
		return deletePoints( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of points by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deletePoints( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * Get the list of points for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the points
	 * @return
	 */
	public List<PointDto> getPoints( String siteUrl )
	{
		return getPoints( PointGetListParams.builder()
			.siteUrl( siteUrl )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a list of points for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of points
	 * @return
	 */
	public List<PointDto> getPoints( PointGetListParams params )
	{
		return getPoints( params, RequestOptions.create() );
	}


	/**
	 * Get a list of points for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of points
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<PointDto> getPoints( PointGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_SITE_URI )
			.responseClass( PointDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of point wallets for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the point wallets
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets( String siteUrl )
	{
		return getPointWallets( PointWalletsGetListParams.builder()
			.siteUrl( siteUrl )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get the list of point wallets for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of point wallets
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets( PointWalletsGetListParams params )
	{
		return getPointWallets( params, RequestOptions.create() );
	}


	/**
	 * Get the list of point wallets for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of point wallets
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public ResponseEntity<PointWalletDto> getPointWallets( PointWalletsGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_SITE_WALLETS_URI )
			.responseClass( PointWalletDto.class )
			.responseType( ResponseType.RESPONSE_ENTITY )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get transactions by Activity ID
	 *
	 * @param activityId
	 *            The ID of the activity for which to retrieve transactions
	 * @return List of PointTransactionDto
	 */
	public List<PointTransactionDto> getTransactionsByActivityId( String activityId )
	{
		return getTransactionsByActivityId( PointTransactionGetByActivityIdParams.builder()
			.activityId( activityId )
			.build() );
	}


	/**
	 * Get transactions by Activity ID
	 *
	 * @param params
	 *            The parameters required to retrieve the transactions
	 * @return List of PointTransactionDto
	 */
	public List<PointTransactionDto> getTransactionsByActivityId( PointTransactionGetByActivityIdParams params )
	{
		return getTransactionsByActivityId( params, RequestOptions.create() );
	}


	/**
	 * Get transactions by Activity ID
	 *
	 * @param params
	 *            The parameters required to retrieve the transactions
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of PointTransactionDto
	 */
	public List<PointTransactionDto> getTransactionsByActivityId(
	    PointTransactionGetByActivityIdParams params,
	    RequestOptions requestOptions)
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_TRANSACTIONS_BY_ACTIVITY_URI )
			.responseClass( PointTransactionDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * This method is used to add custom fields to an existing point.
	 *
	 * @param pointId
	 *            The point to which to add the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto addCustomFields( String pointId, CustomFieldValueRequestData data )
	{
		PointCustomFieldParams params = PointCustomFieldParams.builder().pointId( pointId ).build();
		return modCustomFields( HttpMethod.POST, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to add custom fields to an existing point.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto addCustomFields( PointCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.POST, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to update custom fields of an existing point.
	 *
	 * @param pointId
	 *            The point to update the custom fields of
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto updateCustomFields( String pointId, CustomFieldValueRequestData data )
	{
		PointCustomFieldParams params = PointCustomFieldParams.builder().pointId( pointId ).build();
		return modCustomFields( HttpMethod.PUT, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to update custom fields of an existing point.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto updateCustomFields( PointCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.PUT, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to delete custom fields from an existing point.
	 *
	 * @param pointId
	 *            The point to delete the custom fields from
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto deleteCustomFields( String pointId, CustomFieldValueRequestData data )
	{
		PointCustomFieldParams params = PointCustomFieldParams.builder().pointId( pointId ).build();
		return modCustomFields( HttpMethod.DELETE, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to delete custom fields from an existing point.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public PointDto deleteCustomFields( PointCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, RequestOptions.create() );
	}


	/**
	 * Generic method used to manipulate the custom fields
	 *
	 * @param method
	 *            The HTTP method to use (POST, PUT, DELETE)
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	private PointDto modCustomFields(
		HttpMethod method,
		PointCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_CUSTOM_URI )
			.responseClass( PointDto.class )
			.method( method )
			.params( params )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing point.
	 *
	 * @param pointId
	 *            The ID of the point for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the point
	 * @return
	 */
	public PointDto addCustomField( String pointId, CustomFieldValueDto data, File image )
	{
		return addCustomField( pointId, data, image, RequestOptions.create() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing point.
	 *
	 * @param pointId
	 *            The ID of the point for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto addCustomField(
		String pointId,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		PointCustomFieldParams params = PointCustomFieldParams.builder().pointId( pointId ).build();
		return addCustomField( params, data, image, requestOptions );
	}


	/**
	 * This method is used to add a custom field with an image to an existing point.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the point
	 * @return
	 */
	public PointDto addCustomField( PointCustomFieldParams params, CustomFieldValueDto data, File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing point.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the point
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public PointDto addCustomField(
		PointCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( POINTS_CUSTOM_IMAGE_URI )
			.responseClass( PointDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.requestData( new FileRequestData( image, data ) )
			.build() );
	}


	/**
	 * Return an empty {@link PointRequestData} object
	 *
	 * @return
	 */
	public PointRequestData newPointRequestData()
	{
		return new PointRequestData();
	}
}
