/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.notification;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.notification.data.ClearNotificationsRequestData;
import io.mambo.sdk.service.notification.model.NotificationDto;
import io.mambo.sdk.service.notification.param.NotificationClearParams;
import io.mambo.sdk.service.notification.param.NotificationGetByActivityIdParams;
import io.mambo.sdk.service.notification.param.NotificationGetListParams;
import io.mambo.sdk.service.notification.param.NotificationGetParams;

public class NotificationsService extends AbstractService
{
	private static final String NOTIFICATIONS_URI = "/v1/notifications";
	private static final String NOTIFICATIONS_ACTIVITY_ID_URI = NOTIFICATIONS_URI + "/activity/{activityId}";
	private static final String NOTIFICATIONS_ID_URI = NOTIFICATIONS_URI + "/{notificationId}";
	private static final String CLEAR_NOTIFICATIONS_ID_URI = NOTIFICATIONS_ID_URI + "/clear";
	private static final String CLEAR_NOTIFICATIONS_URI = NOTIFICATIONS_URI + "/clear";
	private static final String NOTIFICATIONS_SITE_URI = "/v1/{siteUrl}/notifications";


	public NotificationsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Get notifications by Activity ID
	 *
	 * @param activityId
	 *            The ID of the activity for which to retrieve notifications
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getByActivityId( String activityId )
	{
		return getByActivityId( NotificationGetByActivityIdParams.builder()
			.activityId( activityId )
			.build() );
	}


	/**
	 * Get notifications by Activity ID
	 *
	 * @param params
	 *            The parameters required to retrieve the notifications
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getByActivityId( NotificationGetByActivityIdParams params )
	{
		return getByActivityId( params, RequestOptions.create() );
	}


	/**
	 * Get notifications by Activity ID
	 *
	 * @param params
	 *            The parameters required to retrieve the notifications
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getByActivityId(
		NotificationGetByActivityIdParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( NOTIFICATIONS_ACTIVITY_ID_URI )
			.responseClass( NotificationDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a notification by ID
	 *
	 * @param notificationId
	 *            The ID of the notification to retrieve
	 * @return NotificationDto
	 */
	public NotificationDto get( String notificationId )
	{
		return get( NotificationGetParams.builder()
			.notificationId( notificationId )
			.build() );
	}


	/**
	 * Get a notification by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the notification
	 * @return NotificationDto
	 */
	public NotificationDto get( NotificationGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a notification by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the notification
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return NotificationDto
	 */
	public NotificationDto get( NotificationGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( NOTIFICATIONS_ID_URI )
			.responseClass( NotificationDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clear a notification by ID
	 *
	 * @param notificationId
	 *            The ID of the notification to clear
	 * @return Status
	 */
	public Status clear( String notificationId )
	{
		return clear( NotificationClearParams.builder()
			.notificationId( notificationId )
			.build() );
	}


	/**
	 * Clear a notification
	 *
	 * @param params
	 *            The parameters required to clear the notification
	 * @return Status
	 */
	public Status clear( NotificationClearParams params )
	{
		return clear( params, RequestOptions.create() );
	}


	/**
	 * Clear a notification
	 *
	 * @param params
	 *            The parameters required to clear the notification
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status clear( NotificationClearParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( CLEAR_NOTIFICATIONS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clear a list of notifications
	 *
	 * @param data
	 *            The data containing the notification IDs to clear
	 * @return Status
	 */
	public Status clear( ClearNotificationsRequestData data )
	{
		return clear( data, RequestOptions.create() );
	}


	/**
	 * Clear a list of notifications
	 *
	 * @param data
	 *            The data containing the notification IDs to clear
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status clear( ClearNotificationsRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( CLEAR_NOTIFICATIONS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * Get the list of notifications for the specified site
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the notifications
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getNotifications( String siteUrl )
	{
		return getNotifications( NotificationGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get the list of notifications for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the notifications
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getNotifications( NotificationGetListParams params )
	{
		return getNotifications( params, RequestOptions.create() );
	}


	/**
	 * Get the list of notifications for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the notifications
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of NotificationDto
	 */
	public List<NotificationDto> getNotifications( NotificationGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( NOTIFICATIONS_SITE_URI )
			.responseClass( NotificationDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}
}