/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.leaderboard.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class LeaderboardGetParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private LeaderboardGetParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		private static final String LEADERBOARD_ID_MUST_NOT_BE_NULL = "leaderboardId must not be null";


		/**
		 * The ID of the leaderboard to retrieve
		 */
		public Builder leaderboardId( String leaderboardId )
		{
			checkNotNull( leaderboardId, LEADERBOARD_ID_MUST_NOT_BE_NULL );
			params.put( "leaderboardId", leaderboardId );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The tag to filter users by
		 */
		public Builder tag( String tag )
		{
			params.put( "tag", tag );
			return this;
		}


		/**
		 * The period for retrieving the leaderboard
		 */
		public Builder period( String period )
		{
			params.put( "period", period );
			return this;
		}


		/**
		 * Whether to include users in response
		 */
		public Builder withUsers( boolean withUsers )
		{
			params.put( "withUsers", String.valueOf( withUsers ) );
			return this;
		}


		/**
		 * User properties to return
		 */
		public Builder userProperties( List<String> userProperties )
		{
			params.putAll( "userProperties", userProperties );
			return this;
		}


		public LeaderboardGetParams build()
		{
			checkContainsKey( params, "leaderboardId", LEADERBOARD_ID_MUST_NOT_BE_NULL );
			return new LeaderboardGetParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
