/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.leaderboard.model;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.common.model.tag.HasTags;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.service.user.model.UserDto;
import io.mambo.sdk.utils.NumberConverter;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to a user in a leaderboard.
 */
@ToString
@EqualsAndHashCode
public class LeaderboardStateDto implements HasTags
{
	private String id;
	private String pointsLastUpdatedOn;
	private String prevRankOn;
	private Long rank;
	private Long prevRank;
	private Number points;
	private Number rankPoints;
	private Number prevRankPoints;
	private UserDto user;
	private List<TagDto> tags;
	private List<String> tagIds;

	/**
	 * This ID uniquely identifies this object
	 */
	public String getId() { return id; }
	public void setId( String id ) { this.id = id; }

	/**
	 * The last date the points were updated on.
	 * This includes increment, decrement and set operations.
	 */
	public String getPointsLastUpdatedOn() { return pointsLastUpdatedOn; }
	public void setPointsLastUpdatedOn( String pointsLastUpdatedOn ) { this.pointsLastUpdatedOn = pointsLastUpdatedOn; }

	/**
	 * The number of points associated to the user in this leaderboard.
	 * See the leaderboard page in administration panel for more information.
	 * @return
	 */
	public Number getPoints() { return points; }
	public void setPoints( Number points ) { this.points = points; }

	@JsonIgnore
	public NumberConverter getPointsHelper() { return new NumberConverter( points ); }

	/**
	 * The rank position of the user
	 * @return
	 */
	public Long getRank() { return rank; }
	public void setRank( Long rank ) { this.rank = rank; }

	/**
	 * The previous rank position of the user (if any)
	 * Note that this will not be set on tag filtered leaderboards
	 * @return
	 */
	public Long getPrevRank() { return prevRank; }
	public void setPrevRank( Long prevRank ) { this.prevRank = prevRank; }

	/**
	 * The number of points the user had when they were ranked
	 * @return
	 */
	public Number getRankPoints() { return rankPoints; }
	public void setRankPoints( Number rankPoints ) { this.rankPoints = rankPoints; }

	@JsonIgnore
	public NumberConverter getRankPointsHelper() { return new NumberConverter( points ); }

	/**
	 * The previous number of points the user had when they were ranked
	 * Note that this will not be set on tag filtered leaderboards
	 * @return
	 */
	public Number getPrevRankPoints() { return prevRankPoints; }
	public void setPrevRankPoints( Number prevRankPoints ) { this.prevRankPoints = prevRankPoints; }

	@JsonIgnore
	public NumberConverter getPrevRankPointsHelper() { return new NumberConverter( points ); }

	/**
	 * The date on which the previous rank was taken.
	 */
	public String getPrevRankOn() { return prevRankOn; }
	public void setPrevRankOn( String prevRankOn ) { this.prevRankOn = prevRankOn; }

	/**
	 * The user associated to the leaderboard.
	 * See the leaderboard page in administration panel for more information.
	 * @return
	 */
	public UserDto getUser() { return user; }
	public void setUser( UserDto user ) { this.user = user; }

	/**
	 * This contains the list of the IDs of the tags which
	 * must be added to the object.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		this.tagIds = tagIds;
	}


	/**
	 * This contains the list of {@link TagDto} associated to this object.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tags; }
	@Override
	public void setTags( List<TagDto> tags ) {
		this.tags = tags;
	}
}
