/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.leaderboard.model;

import java.util.List;

import io.mambo.sdk.service.common.model.interfaces.AggregateDto;
import io.mambo.sdk.service.common.model.interfaces.HasId;
import io.mambo.sdk.service.common.model.tag.HasTags;
import io.mambo.sdk.service.point.model.PointDto;
import io.mambo.sdk.service.tag.model.TagDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to a leaderboard. When the
 * leaderboard is retrieved using the LeaderboardsService getLeaderboards()
 * method, then the users will be null. If instead you retrieve
 * this model using one of the other methods in the LeaderboardsService,
 * then the users will be populated with the top 10 users in the
 * leaderboard.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class LeaderboardDto extends AbstractLeaderboard implements AggregateDto, HasTags, HasId
{
	private Long userCount;
	private String id;
	private String rankedOn;
	private LeaderboardStateDto contextUser;
	private List<LeaderboardStateDto> users;
	private List<PointDto> points;
	private List<TagDto> tags;
	private List<String> tagIds;


	/**
	 * This ID uniquely identifies this object
	 */
	@Override
	public String getId() { return id; }
	public void setId( String id ) { this.id = id; }

	/**
	 * This is the last date on which the ranking positions were updated
	 * @return
	 */
	public String getRankedOn() { return rankedOn; }
	public void setRankedOn(String rankedOn) {
		this.rankedOn = rankedOn;
	}

	/**
	 * When retrieving the leaderboard through the User APIs this field will contain
	 * the user for which we are retrieving the leaderboard. The user may also be found
	 * in the users list if they are a part of the leaderboard users.
	 * Note: this will not be returned when retrieving the list of leader boards.
	 * @return
	 */
	public LeaderboardStateDto getContextUser() { return contextUser; }
	public void setContextUser( LeaderboardStateDto contextUser ) { this.contextUser = contextUser; }

	/**
	 * The list of users in this leaderboard, ordered from 1st to 10th.
	 * See the leaderboard page in administration panel for more information.
	 * @return
	 */
	public List<LeaderboardStateDto> getUsers() { return users; }
	public void setUsers( List<LeaderboardStateDto> users ) { this.users = users; }

	/**
	 * When retrieving the leaderboard with the list of users this will contain
	 * the total number of users who are in the current leaderboard taking into
	 * account any filters applied (i.e. tags).
	 */
	public Long getUserCount() { return userCount; }
	public void setUserCount( Long userCount ) { this.userCount = userCount; }

	/**
	 * The list of points associated to this leaderboard
	 * @return
	 */
	public List<PointDto> getPoints() { return points; }
	public void setPoints( List<PointDto> points ) { this.points = points; }

	/**
	 * This contains the list of the IDs of the tags which
	 * must be added to the object.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		this.tagIds = tagIds;
	}


	/**
	 * This contains the list of {@link TagDto} associated to this object.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tags; }
	@Override
	public void setTags( List<TagDto> tags ) {
		this.tags = tags;
	}
}
