/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.leaderboard.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.leaderboard.data.LeaderboardRequestData;
import io.mambo.sdk.service.leaderboard.model.attributes.LeaderboardAttrs;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Leaderboard data shared by {@link LeaderboardDto} and {@link LeaderboardRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractLeaderboard implements HasTranslation
{
	private static final String POINT_IDS = "pointIds";

	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String startDate;
	private String endDate;
	private Boolean active;
	private Boolean withRanking;
	private Boolean isReverse;
	private Boolean autoFilterByTag;
	private Boolean includeInactiveUsers;
	private Boolean stopAtZero;
	private List<String> pointIds;
	private List<String> filterableByTagIds;
	private List<String> excludedUsersTagIds;
	private List<String> rankBy;
	private LeaderboardAttrs attrs;
	private List<LeaderboardTranslationDto> translations;

	/**
	 * The name of the leaderboard. See the leaderboard page in
	 * administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * Indicates whether the leaderboard is active or not.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * This represents the date from which this leaderboard is active.
	 * If no date is specified, the leaderboard is always active.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getStartDate() { return startDate; }
	public void setStartDate( String startDate ) {
		initializedFields.add( "startDate" );
		this.startDate = startDate;
	}

	/**
	 * This represents the date from which this leaderboard is no longer active.
	 * If no date is specified, the leaderboard is always active.
	 * This must be a timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ.
	 * @return
	 */
	public String getEndDate() { return endDate; }
	public void setEndDate( String endDate ) {
		initializedFields.add( "endDate" );
		this.endDate = endDate;
	}

	/**
	 * Determines which time periods the ranking will take place for.
	 * Only specified periods will be ranked and retrievable via the APIs.
	 * Note that if this property is empty, the leaderboard will be unranked.
	 * Valid values include: day, week, month, all
	 */
	public List<String> getRankBy() { return rankBy; }
	public void setRankBy( List<String> rankBy ) {
		initializedFields.add( "rankBy" );
		this.rankBy = rankBy;
	}

	/**
	 * Indicates whether the leaderboard should be in the reverse order.
	 * This will place the smallest values at the top of the leaderboard.
	 * @return
	 */
	public Boolean getIsReverse() { return isReverse; }
	public void setIsReverse( Boolean isReverse ) {
		initializedFields.add( "isReverse" );
		this.isReverse = isReverse;
	}

	/**
	 * Indicates whether the lowest leaderboard score should be zero or
	 * whether the leaderboard can have a negative score.
	 * @return
	 */
	public Boolean getStopAtZero() { return stopAtZero; }
	public void setStopAtZero( Boolean stopAtZero ) {
		initializedFields.add( "stopAtZero" );
		this.stopAtZero = stopAtZero;
	}

	/**
	 * Indicates whether the leaderboard should be automatically filtered
	 * by the tag even if the tag is not explicitly defined in the API call.
	 * In order for auto filtering to work, there must only be one tag defined
	 * in the filterableByTagIds list. This tag will be used as the one to
	 * auto filter the leaderboard with.
	 * @return
	 */
	public Boolean getAutoFilterByTag() { return autoFilterByTag; }
	public void setAutoFilterByTag( Boolean autoFilterByTag ) {
		initializedFields.add( "autoFilterByTag" );
		this.autoFilterByTag = autoFilterByTag;
	}

	/**
	 * Indicates whether users in the leaderboard should remain in the leaderboard
	 * when their last activity becomes older than the period being tracked.
	 * @return
	 */
	public Boolean getIncludeInactiveUsers() { return includeInactiveUsers; }
	public void setIncludeInactiveUsers( Boolean includeInactiveUsers ) {
		initializedFields.add( "includeInactiveUsers" );
		this.includeInactiveUsers = includeInactiveUsers;
	}

	/**
	 * This contains the list of the tag IDs by which this
	 * leaderboard can be filtered.
	 */
	public List<String> getFilterableByTagIds() { return filterableByTagIds; }
	public void setFilterableByTagIds( List<String> filterableByTagIds ) {
		initializedFields.add( "filterableByTagIds" );
		this.filterableByTagIds = filterableByTagIds;
	}


	/**
	 * This contains the list of the tag IDs such that any user with at least
	 * one of those tags is excluded from the leaderboard.
	 */
	public List<String> getExcludedUsersTagIds() { return excludedUsersTagIds; }
	public void setExcludedUsersTagIds( List<String> excludedUsersTagIds ) {
		initializedFields.add( "excludedUsersTagIds" );
		this.excludedUsersTagIds = excludedUsersTagIds;
	}

	/**
	 * The attributes of the leaderboard. There are currently two types of
	 * attributes: LeaderboardBehaviourAttrs and LeaderboardPointAttrs.
	 * @return
	 */
	public LeaderboardAttrs getAttrs() { return attrs; }
	public void setAttrs( LeaderboardAttrs attrs ) {
		initializedFields.add( "attrs" );
		this.attrs = attrs;
	}

	/**
	 * This must contain the list of the IDs of the points which must
	 * be added together for this leaderboard score.
	 * @return
	 */
	public List<String> getPointIds() { return pointIds; }
	public void setPointIds( List<String> pointIds ) {
		initializedFields.add( POINT_IDS );
		this.pointIds = pointIds;
	}
	public void addPointId( String pointId ) {
		if( pointIds == null ) {
			initializedFields.add( POINT_IDS );
			pointIds = new ArrayList<>();
		}
		pointIds.add( pointId );
	}

	public void addPoint( PointStoreDto point )
	{
		initializedFields.add( POINT_IDS );

		if( pointIds == null ) {
			pointIds = new ArrayList<>();
		}
		pointIds.add( point.getPointId() );
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<LeaderboardTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<LeaderboardTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( LeaderboardTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public LeaderboardTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( LeaderboardTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}
}
