/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.leaderboard;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.leaderboard.data.LeaderboardRequestData;
import io.mambo.sdk.service.leaderboard.model.LeaderboardDto;
import io.mambo.sdk.service.leaderboard.param.LeaderboardCloneParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardCreateParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardDeleteParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardGetListParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardGetParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardRankParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardRegenerateParams;
import io.mambo.sdk.service.leaderboard.param.LeaderboardUpdateParams;

/**
 * The LeaderboardsService class handles all Leaderboard related requests to the
 * Mambo API.
 */
public class LeaderboardsService extends AbstractService
{
	private static final String LEADERBOARDS_URI = "/v1/leaderboards";
	private static final String LEADERBOARDS_ID_URI = LEADERBOARDS_URI + "/{leaderboardId}";
	private static final String LEADERBOARDS_CLONE_URI = LEADERBOARDS_ID_URI + "/clone";
	private static final String LEADERBOARDS_REGENERATE_URI = LEADERBOARDS_ID_URI + "/regenerate";
	private static final String LEADERBOARDS_RANK_URI = LEADERBOARDS_ID_URI + "/rank";
	private static final String LEADERBOARDS_SITE_URI = "/v1/{siteUrl}/leaderboards";


	public LeaderboardsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new leaderboard.
	 */
	public LeaderboardDto create( String siteUrl, LeaderboardRequestData data )
	{
		return create( LeaderboardCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * This method is used to create a new leaderboard.
	 */
	public LeaderboardDto create( String siteUrl, LeaderboardRequestData data, RequestOptions requestOptions )
	{
		return create( LeaderboardCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new leaderboard.
	 */
	public LeaderboardDto create( LeaderboardCreateParams params, LeaderboardRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new leaderboard.
	 */
	public LeaderboardDto create(
		LeaderboardCreateParams params,
		LeaderboardRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_SITE_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing leaderboard by ID.
	 */
	public LeaderboardDto update( String leaderboardId, LeaderboardRequestData data )
	{
		return update( LeaderboardUpdateParams.builder()
			.leaderboardId( leaderboardId )
			.build(), data );
	}


	/**
	 * Update an existing leaderboard by ID.
	 *
	 * @param params
	 *            The parameters required to update the leaderboard
	 * @param data
	 *            The leaderboard data to update
	 * @return
	 */
	public LeaderboardDto update( LeaderboardUpdateParams params, LeaderboardRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing leaderboard by ID.
	 *
	 * @param params
	 *            The parameters required to update the leaderboard
	 * @param data
	 *            The leaderboard data to update
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto update(
		LeaderboardUpdateParams params,
		LeaderboardRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_ID_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a leaderboard by ID.
	 */
	public LeaderboardDto clone( String leaderboardId )
	{
		return clone( LeaderboardCloneParams.builder()
			.leaderboardId( leaderboardId )
			.build() );
	}


	/**
	 * Clone a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to clone the leaderboard
	 * @return
	 */
	public LeaderboardDto clone( LeaderboardCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to clone the leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto clone( LeaderboardCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_CLONE_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Regenerate a leaderboard's score board from the ground up by ID.
	 */
	public LeaderboardDto regenerate( String leaderboardId )
	{
		return regenerate( LeaderboardRegenerateParams.builder()
			.leaderboardId( leaderboardId )
			.build() );
	}


	/**
	 * Regenerate a leaderboard's score board using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to regenerate the leaderboard
	 * @return
	 */
	public LeaderboardDto regenerate( LeaderboardRegenerateParams params )
	{
		return regenerate( params, RequestOptions.create() );
	}


	/**
	 * Regenerate a leaderboard's score board using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to regenerate the leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto regenerate( LeaderboardRegenerateParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_REGENERATE_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Rank a leaderboard's users by ID.
	 */
	public LeaderboardDto rank( String leaderboardId )
	{
		return rank( LeaderboardRankParams.builder()
			.leaderboardId( leaderboardId )
			.build() );
	}


	/**
	 * Rank a leaderboard's users using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to rank the leaderboard
	 * @return
	 */
	public LeaderboardDto rank( LeaderboardRankParams params )
	{
		return rank( params, RequestOptions.create() );
	}


	/**
	 * Rank a leaderboard's users using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to rank the leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto rank( LeaderboardRankParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_RANK_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a leaderboard by ID.
	 */
	public Status delete( String leaderboardId )
	{
		return delete( LeaderboardDeleteParams.builder()
			.leaderboardId( leaderboardId )
			.build() );
	}


	/**
	 * Delete a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to delete the leaderboard
	 * @return
	 */
	public Status delete( LeaderboardDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to delete the leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( LeaderboardDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete multiple leaderboards using the specified data.
	 *
	 * @param data
	 *            The data specifying which leaderboards to delete
	 * @return
	 */
	public Status deleteLeaderboards( DeleteRequestData data )
	{
		return deleteLeaderboards( data, RequestOptions.create() );
	}


	/**
	 * Delete multiple leaderboards using the specified data.
	 *
	 * @param data
	 *            The data specifying which leaderboards to delete
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteLeaderboards( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a leaderboard by ID.
	 */
	public LeaderboardDto get( String leaderboardId )
	{
		return get( LeaderboardGetParams.builder()
			.leaderboardId( leaderboardId )
			.build() );
	}


	/**
	 * Get a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to get the leaderboard
	 * @return
	 */
	public LeaderboardDto get( LeaderboardGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a leaderboard using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to get the leaderboard
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LeaderboardDto get( LeaderboardGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_ID_URI )
			.responseClass( LeaderboardDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get the list of leaderboards for the specified site.
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the leaderboards
	 */
	public List<LeaderboardDto> getLeaderboards( String siteUrl )
	{
		return getLeaderboards( LeaderboardGetListParams.builder().siteUrl( siteUrl ).build() );
	}


	/**
	 * Get a list of leaderboards using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to get the leaderboard list
	 * @return
	 */
	public List<LeaderboardDto> getLeaderboards( LeaderboardGetListParams params )
	{
		return getLeaderboards( params, RequestOptions.create() );
	}


	/**
	 * Get a list of leaderboards using the specified parameters.
	 *
	 * @param params
	 *            The parameters required to get the leaderboard list
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<LeaderboardDto> getLeaderboards( LeaderboardGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LEADERBOARDS_SITE_URI )
			.responseClass( LeaderboardDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return an empty {@link LeaderboardRequestData} object.
	 */
	public LeaderboardRequestData newLeaderboardRequestData()
	{
		return new LeaderboardRequestData();
	}
}