/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.language;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.language.data.LanguageRequestData;
import io.mambo.sdk.service.language.model.LanguageDto;
import io.mambo.sdk.service.language.param.LanguageCloneParams;
import io.mambo.sdk.service.language.param.LanguageCreateParams;
import io.mambo.sdk.service.language.param.LanguageDeleteParams;
import io.mambo.sdk.service.language.param.LanguageGetListParams;
import io.mambo.sdk.service.language.param.LanguageGetParams;
import io.mambo.sdk.service.language.param.LanguageUpdateParams;

public class LanguagesService extends AbstractService
{
	private static final String LANGUAGES_URI = "/v1/languages";
	private static final String LANGUAGES_ID_URI = LANGUAGES_URI + "/{languageId}";
	private static final String LANGUAGES_SITE_URI = "/v1/{siteUrl}/languages";
	private static final String LANGUAGES_CLONE_URI = LANGUAGES_ID_URI + "/clone";


	public LanguagesService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new language.
	 *
	 * @see LanguageRequestData
	 *
	 * @param siteUrl
	 *            The site to which the language belongs to
	 * @param data
	 *            The language request data
	 * @return
	 */
	public LanguageDto create( String siteUrl, LanguageRequestData data )
	{
		return create( LanguageCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new language.
	 *
	 * @see LanguageRequestData
	 *
	 * @param siteUrl
	 *            The site to which the language belongs to
	 * @param data
	 *            The language request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LanguageDto create( String siteUrl, LanguageRequestData data, RequestOptions requestOptions )
	{
		return create( LanguageCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new language.
	 *
	 * @param params
	 *            The parameters required to create the language
	 * @param data
	 *            The language request data
	 * @return
	 */
	public LanguageDto create( LanguageCreateParams params, LanguageRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new language.
	 *
	 * @param params
	 *            The parameters required to create the language
	 * @param data
	 *            The language request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LanguageDto create(
		LanguageCreateParams params,
		LanguageRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_SITE_URI )
			.responseClass( LanguageDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing language by ID.
	 *
	 * @param languageId
	 *            The ID of the language to update
	 * @param data
	 *            The data with which to update the specified language object
	 * @return
	 */
	public LanguageDto update( String languageId, LanguageRequestData data )
	{
		return update( LanguageUpdateParams.builder()
			.languageId( languageId )
			.build(), data );
	}


	/**
	 * Update an existing language by ID.
	 *
	 * @param params
	 *            The parameters required to update the language
	 * @param data
	 *            The data with which to update the specified language object
	 * @return
	 */
	public LanguageDto update( LanguageUpdateParams params, LanguageRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing language by ID.
	 *
	 * @param params
	 *            The parameters required to update the language
	 * @param data
	 *            The data with which to update the specified language object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LanguageDto update(
		LanguageUpdateParams params,
		LanguageRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_ID_URI )
			.responseClass( LanguageDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a language
	 *
	 * @param languageId
	 *            The ID of the language to clone
	 * @return
	 */
	public LanguageDto clone( String languageId )
	{
		return clone( LanguageCloneParams.builder()
			.languageId( languageId )
			.build() );
	}


	/**
	 * Clone a language
	 *
	 * @param params
	 *            The parameters required to clone the language
	 * @return
	 */
	public LanguageDto clone( LanguageCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a language
	 *
	 * @param params
	 *            The parameters required to clone the language
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LanguageDto clone( LanguageCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_CLONE_URI )
			.responseClass( LanguageDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a language by ID.
	 *
	 * @param languageId
	 *            The ID of the language to delete
	 * @return
	 */
	public Status delete( String languageId )
	{
		return delete( LanguageDeleteParams.builder()
			.languageId( languageId )
			.build() );
	}


	/**
	 * Delete a language by ID.
	 *
	 * @param params
	 *            The parameters required to delete the language
	 * @return
	 */
	public Status delete( LanguageDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a language by ID.
	 *
	 * @param params
	 *            The parameters required to delete the language
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( LanguageDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a language by ID
	 *
	 * @param languageId
	 *            The ID of the language to retrieve
	 * @return
	 */
	public LanguageDto get( String languageId )
	{
		return get( LanguageGetParams.builder()
			.languageId( languageId )
			.build() );
	}


	/**
	 * Get a language by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the language
	 * @return
	 */
	public LanguageDto get( LanguageGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a language by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the language
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public LanguageDto get( LanguageGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_ID_URI )
			.responseClass( LanguageDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of languages by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deleteLanguages( DeleteRequestData data )
	{
		return deleteLanguages( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of languages by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteLanguages( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get the list of languages for the specified site
	 *
	 * @param siteUrl
	 *            The site containing the languages
	 * @return
	 */
	public List<LanguageDto> getLanguages( String siteUrl )
	{
		return getLanguages( LanguageGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get a list of languages for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of languages
	 * @return
	 */
	public List<LanguageDto> getLanguages( LanguageGetListParams params )
	{
		return getLanguages( params, RequestOptions.create() );
	}


	/**
	 * Get a list of languages for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of languages
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<LanguageDto> getLanguages( LanguageGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( LANGUAGES_SITE_URI )
			.responseClass( LanguageDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return an empty {@link LanguageRequestData} object
	 *
	 * @return
	 */
	public LanguageRequestData newLanguageRequestData()
	{
		return new LanguageRequestData();
	}
}