/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class KpiGetStatesParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private KpiGetStatesParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		private static final String KPI_ID_MUST_NOT_BE_NULL = "kpiId must not be null";


		/**
		 * The ID of the KPI for which to retrieve states
		 */
		public Builder kpiId( String kpiId )
		{
			checkNotNull( kpiId, KPI_ID_MUST_NOT_BE_NULL );
			params.put( "kpiId", kpiId );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The list of tags to filter by (if any)
		 */
		public Builder tags( List<String> tags )
		{
			params.putAll( "tags", tags );
			return this;
		}


		/**
		 * Whether the tags should return items containing any one of the tags or
		 * whether the tags should return only items containing all of the tags.
		 * Possible values: hasAnyOf / hasAllOf Default: hasAnyOf
		 */
		public Builder tagsJoin( String tagsJoin )
		{
			params.put( "tagsJoin", tagsJoin );
			return this;
		}


		/**
		 * The tagUuid to use to filter the list by personalization tags
		 */
		public Builder tagUuid( String tagUuid )
		{
			params.put( "tagUuid", tagUuid );
			return this;
		}


		/**
		 * The period for which to retrieve the KPI states
		 */
		public Builder period( String period )
		{
			params.put( "period", period );
			return this;
		}


		/**
		 * The year for which to retrieve the KPI states
		 */
		public Builder year( String year )
		{
			params.put( "year", year );
			return this;
		}


		public KpiGetStatesParams build()
		{
			checkContainsKey( params, "kpiId", KPI_ID_MUST_NOT_BE_NULL );
			return new KpiGetStatesParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}
