/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi.model;

import java.util.Map;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.utils.NumberConverter;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to a KPIs tracker
 */
@ToString
@EqualsAndHashCode
public class KpiTrackerDto
{
	private Number target;
	private Number actual;
	private Map<String,Number> points;


	/**
	 * This is the target value for this tracker. The target value
	 * will depend on whether any default targets were set when
	 * defining the KPI or on whether the KPI targets were modified
	 * using the Activities API. It is possible for the target
	 * to be null.
	 * @return
	 */
	public Number getTarget() { return target; }
	public void setTarget( Number target ) { this.target = target; }

	@JsonIgnore
	public NumberConverter getTargetHelper() { return new NumberConverter( target ); }


	/**
	 * This is the actual value for this tracker. This represents
	 * the actual progress made by a User towards this tracker's
	 * target. If no target is specified then this is simply the
	 * progress made for the KPI.
	 * @return
	 */
	public Number getActual() { return actual; }
	public void setActual( Number actual ) { this.actual = actual; }

	@JsonIgnore
	public NumberConverter getActualHelper() { return new NumberConverter( actual ); }


	/**
	 * These are the points that have been currently assigned to
	 * the user. The map includes the pointId vs. the total number
	 * of points with that ID that have been given to the user.
	 * @return
	 */
	public Map<String,Number> getPoints() { return points; }
	public void setPoints( Map<String,Number> points ) { this.points = points; }

	@JsonIgnore
	public Map<String,NumberConverter> getPointsConverter() {
		return NumberConverter.toNumberConverter( points );
	}
}
