/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.PrizeDto;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to a KPIs prizes
 */
@ToString
@EqualsAndHashCode
public class KpiPrizeDto implements HasInitializedData
{
	private transient List<String> initializedFields = new ArrayList<>();

	private String prizeTargetPeriod;
	private String pointsDistributionType;
	private String bonusPointsType;
	private PrizeDto prizes;


	/**
	 * This determines for which of the target time periods the prize
	 * points will be given. This includes both the regular prize points
	 * and the bonus points (if any). The default value is Monthly.
	 * @return
	 */
	public String getPrizeTargetPeriod() { return prizeTargetPeriod; }
	public void setPrizeTargetPeriod( String prizeTargetPeriod ) {
		initializedFields.add( "prizeTargetPeriod" );
		this.prizeTargetPeriod = prizeTargetPeriod;
	}

	/**
	 * This determines how the KPI's prize points will be distributed.
	 * Options include: (1) on_complete: this means the points will be
	 * distributed when the KPI target is hit; (2) on_progress: this means
	 * the points will be distributed based on the progress made towards
	 * achieving the KPI (e.g. 80% progress means 80% of the points will
	 * be distributed)
	 * @return
	 */
	public String getPointsDistributionType() { return pointsDistributionType; }
	public void setPointsDistributionType( String pointsDistributionType ) {
		initializedFields.add( "pointsDistributionType" );
		this.pointsDistributionType = pointsDistributionType;
	}

	/**
	 * This determines whether bonus points will be distributed to those
	 * who achieve more than 100% of the target. The options for the bonus
	 * include: (1) none: no bonus points; (2) percentage: the percentage
	 * over the target is translated into a percentage of the total prize
	 * points available (e.g. 1000 total points available, user hits 120%
	 * of the target, we take 20% of the 1000 points and give them to the
	 * user i.e. an extra 200 points)
	 * @return
	 */
	public String getBonusPointsType() { return bonusPointsType; }
	public void setBonusPointsType( String bonusPointsType ) {
		initializedFields.add( "bonusPointsType" );
		this.bonusPointsType = bonusPointsType;
	}

	/**
	 * The prizes object is used to define what prizes will be unlocked
	 * when the user hits this KPIs targets. Note that the way the points
	 * are distributed can be modified using the prizePointBonus and
	 * pointsDistributionType properties.
	 * @return
	 */
	public PrizeDto getPrizes() { return prizes; }
	public void setPrizes( PrizeDto prizes ) {
		initializedFields.add( "prizes" );
		this.prizes = prizes;
	}

	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
