/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.kpi.data.KpiRequestData;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * KPI data shared by {@link KpiDto} and {@link KpiRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractKpi implements HasTranslation, HasCustomFields
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String description;
	private Boolean active;
	private Boolean hasPrizes;
	private TrackedDto tracked;
	private RedAmberGreenDto redAmberGreen;
	private List<String> assignToUserUUIDs;
	private List<String> assignToUserIds;
	private List<String> assignToUsersByTagId;
	private List<String> suggestedBehaviourIds;
	private List<String> suggestedBehavioursByTagId;
	private List<String> suggestedRewardIds;
	private List<String> suggestedRewardsByTagId;
	private KpiPrizeDto prizes;
	private List<CustomFieldValueDto> customFields;
	private List<KpiTranslationDto> translations;


	/**
	 * The KPI's name
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * The KPI's description
	 * @return
	 */
	public String getDescription() { return description; }
	public void setDescription( String description ) {
		initializedFields.add( "description" );
		this.description = description;
	}

	/**
	 * Indicates whether the KPI is active or not.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * Defines what mechanic is going to be tracked by the KPI
	 * @return
	 */
	public TrackedDto getTracked() { return tracked; }
	public void setTracked( TrackedDto tracked ) {
		initializedFields.add( "tracked" );
		this.tracked = tracked;
	}

	/**
	 * Defines the Red, Amber, Green (RAG) status of the KPI
	 * @return
	 */
	public RedAmberGreenDto getRedAmberGreen() { return redAmberGreen; }
	public void setRedAmberGreen( RedAmberGreenDto redAmberGreen ) {
		initializedFields.add( "redAmberGreen" );
		this.redAmberGreen = redAmberGreen;
	}

	/**
	 * Defines which users should have this KPI assigned to them by UUID
	 * @return
	 */
	public List<String> getAssignToUserUUIDs() { return assignToUserUUIDs; }
	public void setAssignToUserUuids( List<String> assignToUserUUIDs ) {
		initializedFields.add( "assignToUserUUIDs" );
		this.assignToUserUUIDs = assignToUserUUIDs;
	}

	/**
	 * Defines which users should have this KPI assigned to them
	 * @return
	 */
	public List<String> getAssignToUserIds() { return assignToUserIds; }
	public void setAssignToUserIds( List<String> assignToUserIds ) {
		initializedFields.add( "assignToUserIds" );
		this.assignToUserIds = assignToUserIds;
	}

	/**
	 * Defines which users should have this KPI assigned to them
	 * @return
	 */
	public List<String> getAssignToUsersByTagId() { return assignToUsersByTagId; }
	public void setAssignToUsersByTagId( List<String> assignToUsersByTagId ) {
		initializedFields.add( "assignToUsersByTagId" );
		this.assignToUsersByTagId = assignToUsersByTagId;
	}

	/**
	 * Defines if there are any suggested behaviours associated to this KPI.
	 * A suggested behaviour would be a behaviour that the users can complete
	 * in order to progress with the KPI.
	 * @return
	 */
	public List<String> getSuggestedBehaviourIds() { return suggestedBehaviourIds; }
	public void setSuggestedBehaviourIds( List<String> suggestedBehaviourIds ) {
		initializedFields.add( "suggestedBehaviourIds" );
		this.suggestedBehaviourIds = suggestedBehaviourIds;
	}

	/**
	 * Defines if there are any suggested behaviours by tag associated to this KPI.
	 * A suggested behaviour would be a behaviour that the users can complete
	 * in order to progress with the KPI.
	 * @return
	 */
	public List<String> getSuggestedBehavioursByTagId() { return suggestedBehavioursByTagId; }
	public void setSuggestedBehavioursByTagId( List<String> suggestedBehavioursByTagId ) {
		initializedFields.add( "suggestedBehavioursByTagId" );
		this.suggestedBehavioursByTagId = suggestedBehavioursByTagId;
	}

	/**
	 * Defines if there are any suggested rewards associated to this KPI.
	 * A suggested reward would be a reward that the users can complete
	 * in order to progress with the KPI.
	 * @return
	 */
	public List<String> getSuggestedRewardIds() { return suggestedRewardIds; }
	public void setSuggestedRewardIds( List<String> suggestedRewardIds ) {
		initializedFields.add( "suggestedRewardIds" );
		this.suggestedRewardIds = suggestedRewardIds;
	}

	/**
	 * Defines if there are any suggested rewards by tag associated to this KPI.
	 * A suggested reward would be a reward that the users can complete
	 * in order to progress with the KPI.
	 * @return
	 */
	public List<String> getSuggestedRewardsByTagId() { return suggestedRewardsByTagId; }
	public void setSuggestedRewardsByTagId( List<String> suggestedRewardsByTagId ) {
		initializedFields.add( "suggestedRewardsByTagId" );
		this.suggestedRewardsByTagId = suggestedRewardsByTagId;
	}

	/**
	 * Indicates whether the KPI will distribute prizes when targets are met.
	 * @return
	 */
	public Boolean getHasPrizes() { return hasPrizes; }
	public void setHasPrizes( Boolean hasPrizes ) {
		initializedFields.add( "hasPrizes" );
		this.hasPrizes = hasPrizes;
	}

	/**
	 * The prizes object is used to define what prizes will be unlocked
	 * when the user hits this KPIs targets. Note that the way the points
	 * are distributed can be modified using the prizePointBonus and
	 * pointsDistributionType properties.
	 * @return
	 */
	public KpiPrizeDto getPrizes() { return prizes; }
	public void setPrizes( KpiPrizeDto prizes ) {
		initializedFields.add( "prizes" );
		this.prizes = prizes;
	}

	/**
	 * Custom fields defined for the KPI. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the KPI.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<KpiTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<KpiTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( KpiTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public KpiTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( KpiTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}
}
