/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi.data;

import java.util.List;

import io.mambo.sdk.service.common.data.RequestData;
import io.mambo.sdk.service.common.model.tag.HasTagsRequestData;
import io.mambo.sdk.service.kpi.model.AbstractKpi;
import io.mambo.sdk.service.tag.model.TagDto;
import io.mambo.sdk.utils.TagUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This object captures the data required by the KPI API in
 * order to create and update KPIs.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class KpiRequestData extends AbstractKpi implements RequestData, HasTagsRequestData
{
	private Boolean removeImage;
	private List<TagDto> tagDtos;
	private List<String> tagIds;
	private List<String> tagTags;


	/**
	 * Whether the image associated to the Reward should be removed
	 * @return
	 */
	public Boolean getRemoveImage() { return removeImage; }
	public void setRemoveImage( Boolean removeImage ) {
		initializedFields.add( "removeImage" );
		this.removeImage = removeImage;
	}


	/**
	 * This should contain the list of the IDs of the tags which
	 * must be added to the object.
	 * @return
	 */
	@Override
	public List<String> getTagIds() { return tagIds; }
	@Override
	public void setTagIds( List<String> tagIds ) {
		initializedFields.add( "tagIds" );
		this.tagIds = tagIds;
	}


	/**
	 * This contains the list of {@link TagDto} associated to this object.
	 * @return
	 */
	@Override
	public List<TagDto> getTags() { return tagDtos; }
	@Override
	public void setTags( List<TagDto> tagDtos ) {
		// DO not call initializedFields here as we
		// don't want to send the TagDtos to the server
		setTagIds( TagUtils.extractTagIds( tagDtos ) );
		this.tagDtos = tagDtos;
	}


	/**
	 * This should contain the list of the tags which must be added to the object.
	 * This method accepts the actual tag value, for example: hr_dept, finance_dept, etc.
	 * @return
	 */
	@Override
	public List<String> getTagTags() { return tagTags; }
	@Override
	public void setTagTags( List<String> tagTags ) {
		initializedFields.add( "tagTags" );
		this.tagTags = tagTags;
	}


	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}