/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.kpi;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.kpi.data.KpiRequestData;
import io.mambo.sdk.service.kpi.data.KpiTargetRequestData;
import io.mambo.sdk.service.kpi.model.KpiDto;
import io.mambo.sdk.service.kpi.model.KpiStateDto;
import io.mambo.sdk.service.kpi.model.KpiTargetDto;
import io.mambo.sdk.service.kpi.param.KpiCloneParams;
import io.mambo.sdk.service.kpi.param.KpiCreateParams;
import io.mambo.sdk.service.kpi.param.KpiCustomFieldParams;
import io.mambo.sdk.service.kpi.param.KpiDeleteParams;
import io.mambo.sdk.service.kpi.param.KpiGetListParams;
import io.mambo.sdk.service.kpi.param.KpiGetParams;
import io.mambo.sdk.service.kpi.param.KpiGetStatesByUserParams;
import io.mambo.sdk.service.kpi.param.KpiGetStatesParams;
import io.mambo.sdk.service.kpi.param.KpiTargetCreateParams;
import io.mambo.sdk.service.kpi.param.KpiTargetGetListParams;
import io.mambo.sdk.service.kpi.param.KpiTargetUpdateParams;
import io.mambo.sdk.service.kpi.param.KpiUpdateParams;

public class KpisService extends AbstractService
{
	private static final String KPIS_URI = "/v1/kpis";
	private static final String KPIS_ID_URI = KPIS_URI + "/{kpiId}";
	private static final String KPIS_IMAGE_URI = KPIS_ID_URI + "/image";
	private static final String KPIS_CLONE_URI = KPIS_ID_URI + "/clone";
	private static final String KPIS_CUSTOM_URI = KPIS_ID_URI + "/custom_fields";
	private static final String KPIS_CUSTOM_IMAGE_URI = KPIS_ID_URI + "/custom_fields/image";
	private static final String KPIS_STATES_URI = KPIS_ID_URI + "/states";
	private static final String KPIS_TARGETS_URI = KPIS_ID_URI + "/targets";
	private static final String KPIS_TARGETS_ID_URI = KPIS_URI + "/targets/{targetId}";
	private static final String KPIS_TARGETS_NO_ID_URI = KPIS_URI + "/targets";
	private static final String KPIS_SITE_URI = "/v1/{siteUrl}/kpis";
	private static final String KPIS_UUID_URI = KPIS_SITE_URI + "/{uuid}";


	public KpisService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new KPI.
	 *
	 * @param siteUrl
	 *            The URL of the site in which to create the KPI
	 * @param data
	 *            The KPI request data
	 * @return
	 */
	public KpiDto create( String siteUrl, KpiRequestData data )
	{
		return create( KpiCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new KPI.
	 *
	 * @param siteUrl
	 *            The URL of the site in which to create the KPI
	 * @param data
	 *            The KPI request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto create( String siteUrl, KpiRequestData data, RequestOptions requestOptions )
	{
		return create( KpiCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new KPI.
	 *
	 * @param params
	 *            The parameters required to create the KPI
	 * @param data
	 *            The KPI request data
	 * @return
	 */
	public KpiDto create( KpiCreateParams params, KpiRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new KPI.
	 *
	 * @param params
	 *            The parameters required to create the KPI
	 * @param data
	 *            The KPI request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto create(
		KpiCreateParams params,
		KpiRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_SITE_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Update an existing KPI by ID.
	 *
	 * @param kpiId
	 *            The ID of the KPI to update
	 * @param data
	 *            The data with which to update the specified KPI object
	 * @return
	 */
	public KpiDto update( String kpiId, KpiRequestData data )
	{
		return update( KpiUpdateParams.builder()
			.kpiId( kpiId )
			.build(), data );
	}


	/**
	 * Update an existing KPI
	 *
	 * @param params
	 *            The parameters required to update the KPI
	 * @param data
	 *            The data with which to update the specified KPI
	 * @return
	 */
	public KpiDto update( KpiUpdateParams params, KpiRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing KPI
	 *
	 * @param params
	 *            The parameters required to update the KPI
	 * @param data
	 *            The data with which to update the specified KPI
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto update(
		KpiUpdateParams params,
		KpiRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_ID_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Upload an image for the KPI
	 *
	 * @param kpiId
	 *            The KPI for which to upload the image
	 * @param image
	 *            The image to upload
	 * @return
	 */
	public KpiDto uploadImage( String kpiId, File image )
	{
		return uploadImage( KpiUpdateParams.builder().kpiId( kpiId ).build(), image );
	}


	/**
	 * Upload an image for the KPI
	 *
	 * @param params
	 *            The parameters required to upload the image
	 * @param image
	 *            The image to upload
	 * @return
	 */
	public KpiDto uploadImage( KpiUpdateParams params, File image )
	{
		return uploadImage( params, image, RequestOptions.create() );
	}


	/**
	 * Upload an image for the KPI
	 *
	 * @param params
	 *            The parameters required to upload the image
	 * @param image
	 *            The image to upload
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto uploadImage( KpiUpdateParams params, File image, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_IMAGE_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image ) )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Clone a KPI
	 *
	 * @param kpiId
	 *            The ID of the KPI to clone
	 * @return
	 */
	public KpiDto clone( String kpiId )
	{
		return clone( KpiCloneParams.builder()
			.kpiId( kpiId )
			.build() );
	}


	/**
	 * Clone a KPI
	 *
	 * @param params
	 *            The parameters required to clone the KPI
	 * @return
	 */
	public KpiDto clone( KpiCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a KPI
	 *
	 * @param params
	 *            The parameters required to clone the KPI
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto clone( KpiCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_CLONE_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Delete a KPI by ID.
	 *
	 * @param kpiId
	 *            The ID of the KPI to delete
	 * @return
	 */
	public Status delete( String kpiId )
	{
		return delete( KpiDeleteParams.builder()
			.kpiId( kpiId )
			.build() );
	}


	/**
	 * Delete a KPI
	 *
	 * @param params
	 *            The parameters required to delete the KPI
	 * @return
	 */
	public Status delete( KpiDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a KPI
	 *
	 * @param params
	 *            The parameters required to delete the KPI
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( KpiDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a KPI by ID
	 *
	 * @param kpiId
	 *            The ID of the KPI to retrieve
	 * @return
	 */
	public KpiDto get( String kpiId )
	{
		return get( KpiGetParams.builder()
			.kpiId( kpiId )
			.build() );
	}


	/**
	 * Get a KPI
	 *
	 * @param params
	 *            The parameters required to retrieve the KPI
	 * @return
	 */
	public KpiDto get( KpiGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a KPI
	 *
	 * @param params
	 *            The parameters required to retrieve the KPI
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto get( KpiGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_ID_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Delete a list of KPIs
	 *
	 * @param data
	 *            The DeleteRequestData containing the KPIs to delete
	 * @return
	 */
	public Status deleteKpis( DeleteRequestData data )
	{
		return deleteKpis( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of KPIs
	 *
	 * @param data
	 *            The DeleteRequestData containing the KPIs to delete
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteKpis( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a list of KPIs
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the KPIs
	 * @return
	 */
	public List<KpiDto> getKpis( String siteUrl )
	{
		return getKpis( KpiGetListParams.builder()
			.siteUrl( siteUrl )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a list of KPIs
	 *
	 * @param params
	 *            The parameters required to retrieve the list
	 * @return
	 */
	public List<KpiDto> getKpis( KpiGetListParams params )
	{
		return getKpis( params, RequestOptions.create() );
	}


	/**
	 * Get a list of KPIs
	 *
	 * @param params
	 *            The parameters required to retrieve the list
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<KpiDto> getKpis( KpiGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_SITE_URI )
			.responseClass( KpiDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get states for a KPI
	 *
	 * @param kpiId
	 *            The KPI for which to retrieve the KPI states
	 * @return
	 */
	public List<KpiStateDto> getKpiStates( String kpiId )
	{
		return getKpiStates( KpiGetStatesParams.builder()
			.kpiId( kpiId )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get states for a KPI
	 *
	 * @param params
	 *            The parameters required to retrieve KPI states
	 * @return
	 */
	public List<KpiStateDto> getKpiStates( KpiGetStatesParams params )
	{
		return getKpiStates( params, RequestOptions.create() );
	}


	/**
	 * Get states for a KPI
	 *
	 * @param params
	 *            The parameters required to retrieve KPI states
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<KpiStateDto> getKpiStates( KpiGetStatesParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_STATES_URI )
			.responseClass( KpiStateDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get states for a KPI
	 *
	 * @param siteUrl
	 *            The site to which the KPI and user belong
	 * @param uuid
	 *            The user for which to retrieve the KPI states
	 * @return
	 */
	public List<KpiStateDto> getKpiStatesByUser( String siteUrl, String uuid )
	{
		return getKpiStatesByUser( KpiGetStatesByUserParams.builder()
			.siteUrl( siteUrl )
			.uuid( uuid )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get states for a KPI
	 *
	 * @param params
	 *            The parameters required to retrieve KPI states
	 * @return
	 */
	public List<KpiStateDto> getKpiStatesByUser( KpiGetStatesByUserParams params )
	{
		return getKpiStatesByUser( params, RequestOptions.create() );
	}


	/**
	 * Get KPI states for a user
	 *
	 * @param params
	 *            The parameters required to retrieve KPI states
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<KpiStateDto> getKpiStatesByUser( KpiGetStatesByUserParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_UUID_URI )
			.responseClass( KpiStateDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Create a new KPI target
	 *
	 * @param kpiId
	 *            The ID of the KPI for which to create the target
	 * @param data
	 *            The target request data
	 * @return
	 */
	public KpiTargetDto createKpiTarget(
		String kpiId,
		KpiTargetRequestData data )
	{
		return createKpiTarget( KpiTargetCreateParams
			.builder()
			.kpiId( kpiId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Create a new KPI target
	 *
	 * @param kpiId
	 *            The ID of the KPI for which to create the target
	 * @param data
	 *            The target request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiTargetDto createKpiTarget(
		String kpiId,
		KpiTargetRequestData data,
		RequestOptions requestOptions )
	{
		return createKpiTarget( KpiTargetCreateParams
			.builder()
			.kpiId( kpiId )
			.build(), data, requestOptions );
	}


	/**
	 * Create a new KPI target
	 *
	 * @param params
	 *            The parameters required to create the target
	 * @param data
	 *            The target request data
	 * @return
	 */
	public KpiTargetDto createKpiTarget(
		KpiTargetCreateParams params,
		KpiTargetRequestData data )
	{
		return createKpiTarget( params, data, RequestOptions.create() );
	}


	/**
	 * Create a new KPI target
	 *
	 * @param params
	 *            The parameters required to create the target
	 * @param data
	 *            The target request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiTargetDto createKpiTarget(
		KpiTargetCreateParams params,
		KpiTargetRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_TARGETS_URI )
			.responseClass( KpiTargetDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Update an existing KPI target by ID.
	 *
	 * @param targetId
	 *            The ID of the target to update
	 * @param data
	 *            The data with which to update the specified target object
	 * @return
	 */
	public KpiTargetDto updateKpiTarget( String targetId, KpiTargetRequestData data )
	{
		return updateKpiTarget( KpiTargetUpdateParams.builder()
			.targetId( targetId )
			.build(), data );
	}


	/**
	 * Update a KPI target
	 *
	 * @param params
	 *            The parameters required to update the target
	 * @param data
	 *            The target request data
	 * @return
	 */
	public KpiTargetDto updateKpiTarget(
		KpiTargetUpdateParams params,
		KpiTargetRequestData data )
	{
		return updateKpiTarget( params, data, RequestOptions.create() );
	}


	/**
	 * Update a KPI target
	 *
	 * @param params
	 *            The parameters required to update the target
	 * @param data
	 *            The target request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiTargetDto updateKpiTarget(
		KpiTargetUpdateParams params,
		KpiTargetRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_TARGETS_ID_URI )
			.responseClass( KpiTargetDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Delete KPI targets
	 *
	 * @param data
	 *            The DeleteRequestData containing the targets to delete
	 * @return
	 */
	public Status deleteKpiTargets( DeleteRequestData data )
	{
		return deleteKpiTargets( data, RequestOptions.create() );
	}


	/**
	 * Delete KPI targets
	 *
	 * @param data
	 *            The DeleteRequestData containing the targets to delete
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteKpiTargets( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_TARGETS_NO_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a list of targets associated to the specified KPI
	 *
	 * @param kpiId
	 *            The KPI containing the targets to use
	 * @return
	 */
	public List<KpiTargetDto> getKpiTargets( String kpiId )
	{
		return getKpiTargets( KpiTargetGetListParams.builder()
			.kpiId( kpiId )
			.build() );
	}


	/**
	 * Get KPI targets
	 *
	 * @param params
	 *            The parameters required to retrieve the targets
	 * @return
	 */
	public List<KpiTargetDto> getKpiTargets( KpiTargetGetListParams params )
	{
		return getKpiTargets( params, RequestOptions.create() );
	}


	/**
	 * Get KPI targets
	 *
	 * @param params
	 *            The parameters required to retrieve the targets
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<KpiTargetDto> getKpiTargets(
		KpiTargetGetListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_TARGETS_URI )
			.responseClass( KpiTargetDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Add custom fields to a KPI
	 *
	 * @param kpiId
	 *            The KPI to which to the custom fields should be added
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto addCustomFields(
		String kpiId,
		CustomFieldValueRequestData data )
	{
		return addCustomFields( KpiCustomFieldParams.builder()
			.kpiId( kpiId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Add custom fields to a KPI
	 *
	 * @param params
	 *            The parameters required to add custom fields
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto addCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return addCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * Add custom fields to a KPI
	 *
	 * @param params
	 *            The parameters required to add custom fields
	 * @param data
	 *            The custom field data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto addCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.POST, params, data, requestOptions );
	}


	/**
	 * Update custom fields of a KPI
	 *
	 * @param kpiId
	 *            The KPI for which the custom fields should be updated
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto updateCustomFields(
		String kpiId,
		CustomFieldValueRequestData data )
	{
		return updateCustomFields( KpiCustomFieldParams.builder()
			.kpiId( kpiId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Update custom fields of a KPI
	 *
	 * @param params
	 *            The parameters required to update custom fields
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto updateCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return updateCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * Update custom fields of a KPI
	 *
	 * @param params
	 *            The parameters required to update custom fields
	 * @param data
	 *            The custom field data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto updateCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.PUT, params, data, requestOptions );
	}


	/**
	 * Delete custom fields from a KPI
	 *
	 * @param kpiId
	 *            The KPI from which the custom fields should be deleted
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto deleteCustomFields(
		String kpiId,
		CustomFieldValueRequestData data )
	{
		return deleteCustomFields( KpiCustomFieldParams.builder()
			.kpiId( kpiId )
			.build(), data, RequestOptions.create() );
	}


	/**
	 * Delete custom fields from a KPI
	 *
	 * @param params
	 *            The parameters required to delete custom fields
	 * @param data
	 *            The custom field data
	 * @return
	 */
	public KpiDto deleteCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data )
	{
		return deleteCustomFields( params, data, RequestOptions.create() );
	}


	/**
	 * Delete custom fields from a KPI
	 *
	 * @param params
	 *            The parameters required to delete custom fields
	 * @param data
	 *            The custom field data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto deleteCustomFields(
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, requestOptions );
	}


	/**
	 * Generic method for custom field operations
	 */
	private KpiDto modCustomFields(
		HttpMethod method,
		KpiCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_CUSTOM_URI )
			.responseClass( KpiDto.class )
			.method( method )
			.requestData( data )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Add a custom field with an image to a KPI
	 *
	 * @param params
	 *            The parameters required to add the custom field
	 * @param data
	 *            The custom field data
	 * @param image
	 *            The image file
	 * @return
	 */
	public KpiDto addCustomField(
		KpiCustomFieldParams params,
		CustomFieldValueDto data,
		File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * Add a custom field with an image to a KPI
	 *
	 * @param params
	 *            The parameters required to add the custom field
	 * @param data
	 *            The custom field data
	 * @param image
	 *            The image file
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public KpiDto addCustomField(
		KpiCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( KPIS_CUSTOM_IMAGE_URI )
			.responseClass( KpiDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image, data ) )
			.params( params )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Return an empty KPIRequestData object
	 *
	 * @return
	 */
	public KpiRequestData newKpiRequestData()
	{
		return new KpiRequestData();
	}


	/**
	 * Return an empty KPITargetRequestData object
	 *
	 * @return
	 */
	public KpiTargetRequestData newKpiTargetRequestData()
	{
		return new KpiTargetRequestData();
	}
}