/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.datastore.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.Order;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class DataStoreGetListParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private DataStoreGetListParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		/**
		 * The site from which to retrieve the list of data stores
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, "siteUrl must not be null" );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The type of data store to retrieve
		 */
		public Builder type( String type )
		{
			params.put( "type", type );
			return this;
		}


		/**
		 * The ownedBy IDs by which to filter the data stores retrieved
		 */
		public Builder ownedByIds( List<String> ownedByIds )
		{
			params.putAll( "ownedByIds", ownedByIds );
			return this;
		}


		/**
		 * The ownedBy UUIDs by which to filter the data stores retrieved
		 */
		public Builder ownedByUuids( List<String> ownedByUuids )
		{
			params.putAll( "ownedByUuids", ownedByUuids );
			return this;
		}


		/**
		 * The user IDs by which to filter the data stores retrieved
		 */
		public Builder userIds( List<String> userIds )
		{
			params.putAll( "userIds", userIds );
			return this;
		}


		/**
		 * The user UUIDs by which to filter the data stores retrieved
		 */
		public Builder userUuids( List<String> userUuids )
		{
			params.putAll( "userUuids", userUuids );
			return this;
		}


		/**
		 * Specifies the start date of the data stores retrieved
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Specifies the end date of the data stores retrieved
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * Specifies the order in which to sort the results
		 */
		public Builder order( Order order )
		{
			params.put( "order", order.value() );
			return this;
		}


		/**
		 * Specifies whether to include system data stores
		 */
		public Builder includeSystemDataStores( boolean includeSystemDataStores )
		{
			params.put( "includeSystemDataStores", String.valueOf( includeSystemDataStores ) );
			return this;
		}


		/**
		 * Specifies whether to return data stores with full user objects
		 */
		public Builder withUsers( boolean withUsers )
		{
			params.put( "withUsers", String.valueOf( withUsers ) );
			return this;
		}


		/**
		 * The list of user properties to be included
		 */
		public Builder userProperties( List<String> userProperties )
		{
			params.putAll( "userProperties", userProperties );
			return this;
		}


		public DataStoreGetListParams build()
		{
			checkContainsKey( params, "siteUrl", "siteUrl must not be null" );
			return new DataStoreGetListParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}