/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.datastore.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

public class DataStoreDeleteByCriteriaParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private DataStoreDeleteByCriteriaParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		/**
		 * The site from which to delete the data stores
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, "siteUrl must not be null" );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * The type of data stores to delete
		 */
		public Builder type( String type )
		{
			params.put( "type", type );
			return this;
		}


		/**
		 * The ownedBy IDs by which to delete the data stores
		 */
		public Builder ownedByIds( List<String> ownedByIds )
		{
			params.putAll( "ownedByIds", ownedByIds );
			return this;
		}


		/**
		 * The ownedBy UUIDs by which to delete the data stores
		 */
		public Builder ownedByUuids( List<String> ownedByUuids )
		{
			params.putAll( "ownedByUuids", ownedByUuids );
			return this;
		}


		/**
		 * Specifies the start date from which to delete data stores
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Specifies the end date until which to delete data stores
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * Specifies whether to include system data stores
		 */
		public Builder includeSystemDataStores( boolean includeSystemDataStores )
		{
			params.put( "includeSystemDataStores", String.valueOf( includeSystemDataStores ) );
			return this;
		}


		public DataStoreDeleteByCriteriaParams build()
		{
			checkContainsKey( params, "siteUrl", "siteUrl must not be null" );
			return new DataStoreDeleteByCriteriaParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}