/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.datastore;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.datastore.data.DataStoreRequestData;
import io.mambo.sdk.service.datastore.model.DataStoreDto;
import io.mambo.sdk.service.datastore.param.DataStoreCloneParams;
import io.mambo.sdk.service.datastore.param.DataStoreCreateParams;
import io.mambo.sdk.service.datastore.param.DataStoreDeleteByCriteriaParams;
import io.mambo.sdk.service.datastore.param.DataStoreDeleteParams;
import io.mambo.sdk.service.datastore.param.DataStoreGetListParams;
import io.mambo.sdk.service.datastore.param.DataStoreGetParams;
import io.mambo.sdk.service.datastore.param.DataStoreUpdateParams;

/**
 * The DataStoresService class handles all DataStore related requests to the
 * Mambo API.
 */
public class DataStoresService extends AbstractService
{
	private static final String DATA_STORES_URI = "/v1/data_stores";
	private static final String DATA_STORES_ID_URI = DATA_STORES_URI + "/{dataStoreId}";
	private static final String DATA_STORES_CLONE_URI = DATA_STORES_ID_URI + "/clone";
	private static final String DATA_STORES_SITE_URI = "/v1/{siteUrl}/data_stores";


	public DataStoresService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new data store.
	 *
	 * @see DataStoreRequestData
	 *
	 * @param siteUrl
	 *            The site to which the data store belongs to
	 * @param data
	 *            The data store request data
	 * @return DataStoreDto
	 */
	public DataStoreDto create( String siteUrl, DataStoreRequestData data )
	{
		return create( DataStoreCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * This method is used to create a new data store.
	 *
	 * @see DataStoreRequestData
	 *
	 * @param siteUrl
	 *            The site to which the data store belongs to
	 * @param data
	 *            The data store request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return DataStoreDto
	 */
	public DataStoreDto create(
		String siteUrl,
		DataStoreRequestData data,
		RequestOptions requestOptions )
	{
		return create( DataStoreCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new data store.
	 *
	 * @param params
	 *            The parameters required to create the data store
	 * @param data
	 *            The data store request data
	 * @return DataStoreDto
	 */
	public DataStoreDto create( DataStoreCreateParams params, DataStoreRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new data store.
	 *
	 * @param params
	 *            The parameters required to create the data store
	 * @param data
	 *            The data store request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return DataStoreDto
	 */
	public DataStoreDto create(
		DataStoreCreateParams params,
		DataStoreRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_SITE_URI )
			.responseClass( DataStoreDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing data store by ID.
	 *
	 * @param dataStoreId
	 *            The ID of the data store to update
	 * @param data
	 *            The data with which to update the specified data store object
	 * @return DataStoreDto
	 */
	public DataStoreDto update( String dataStoreId, DataStoreRequestData data )
	{
		return update( DataStoreUpdateParams.builder()
			.dataStoreId( dataStoreId )
			.build(), data );
	}


	/**
	 * Update an existing data store by ID.
	 *
	 * @param params
	 *            The parameters required to update the data store
	 * @param data
	 *            The data with which to update the specified data store object
	 * @return DataStoreDto
	 */
	public DataStoreDto update( DataStoreUpdateParams params, DataStoreRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing data store by ID.
	 *
	 * @param params
	 *            The parameters required to update the data store
	 * @param data
	 *            The data with which to update the specified data store object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return DataStoreDto
	 */
	public DataStoreDto update(
		DataStoreUpdateParams params,
		DataStoreRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_ID_URI )
			.responseClass( DataStoreDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a data store
	 *
	 * @param dataStoreId
	 *            The ID of the data store to clone
	 * @return DataStoreDto
	 */
	public DataStoreDto clone( String dataStoreId )
	{
		return clone( DataStoreCloneParams.builder()
			.dataStoreId( dataStoreId )
			.build() );
	}


	/**
	 * Clone a data store
	 *
	 * @param params
	 *            The parameters required to clone the data store
	 * @return DataStoreDto
	 */
	public DataStoreDto clone( DataStoreCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a data store
	 *
	 * @param params
	 *            The parameters required to clone the data store
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return DataStoreDto
	 */
	public DataStoreDto clone( DataStoreCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_CLONE_URI )
			.responseClass( DataStoreDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a data store by ID.
	 *
	 * @param dataStoreId
	 *            The ID of the data store to delete
	 * @return Status
	 */
	public Status delete( String dataStoreId )
	{
		return delete( dataStoreId, false );
	}


	/**
	 * Delete a data store by ID.
	 *
	 * @param dataStoreId
	 *            The ID of the data store to delete
	 * @param includeSystemDataStores
	 *            Whether system data stores should be deleted
	 * @return Status
	 */
	public Status delete( String dataStoreId, boolean includeSystemDataStores )
	{
		return delete( DataStoreDeleteParams.builder()
			.dataStoreId( dataStoreId )
			.includeSystemDataStores( includeSystemDataStores )
			.build() );
	}


	/**
	 * Delete a data store by ID.
	 *
	 * @param params
	 *            The parameters required to delete the data store
	 * @return Status
	 */
	public Status delete( DataStoreDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a data store by ID.
	 *
	 * @param params
	 *            The parameters required to delete the data store
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status delete( DataStoreDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a data store by ID
	 *
	 * @param dataStoreId
	 *            The ID of the data store to retrieve
	 * @return DataStoreDto
	 */
	public DataStoreDto get( String dataStoreId )
	{
		return get( DataStoreGetParams.builder()
			.dataStoreId( dataStoreId )
			.build() );
	}


	/**
	 * Get a data store by ID with options
	 *
	 * @param params
	 *            The parameters required to retrieve the data store
	 * @return DataStoreDto
	 */
	public DataStoreDto get( DataStoreGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a data store by ID with options
	 *
	 * @param params
	 *            The parameters required to retrieve the data store
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return DataStoreDto
	 */
	public DataStoreDto get( DataStoreGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_ID_URI )
			.responseClass( DataStoreDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a list of data stores by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return Status
	 */
	public Status deleteDataStores( DeleteRequestData data )
	{
		return deleteDataStores( data, false );
	}


	/**
	 * Delete a list of data stores by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status deleteDataStores( DeleteRequestData data, RequestOptions requestOptions )
	{
		return deleteDataStores( data, false, requestOptions );
	}


	/**
	 * Delete a list of data stores by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param includeSystemDataStores
	 *            Whether system data stores should be deleted
	 * @return Status
	 */
	public Status deleteDataStores( DeleteRequestData data, boolean includeSystemDataStores )
	{
		return deleteDataStores( data, includeSystemDataStores, RequestOptions.create() );
	}


	/**
	 * Delete a list of data stores by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param includeSystemDataStores
	 *            Whether system data stores should be deleted
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status deleteDataStores(
		DeleteRequestData data,
		boolean includeSystemDataStores,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.params( DataStoreDeleteParams.builder()
				.includeSystemDataStores( includeSystemDataStores )
				.build() )
			.build() );
	}


	/**
	 * Delete data stores by criteria
	 *
	 * @param params
	 *            The parameters required to delete the data stores
	 * @return Status
	 */
	public Status deleteDataStoresByCriteria( DataStoreDeleteByCriteriaParams params )
	{
		return deleteDataStoresByCriteria( params, RequestOptions.create() );
	}


	/**
	 * Delete data stores by criteria
	 *
	 * @param params
	 *            The parameters required to delete the data stores
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return Status
	 */
	public Status deleteDataStoresByCriteria(
		DataStoreDeleteByCriteriaParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_SITE_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a list of data stores for the specified site.
	 *
	 * @param siteUrl
	 *            The site from which to retrieve the data stores
	 * @return List of DataStoreDto
	 */
	public List<DataStoreDto> getDataStores( String siteUrl )
	{
		return getDataStores( DataStoreGetListParams.builder()
			.siteUrl( siteUrl )
			.build(), RequestOptions.create() );
	}


	/**
	 * Get a list of data stores for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of data stores
	 * @return List of DataStoreDto
	 */
	public List<DataStoreDto> getDataStores( DataStoreGetListParams params )
	{
		return getDataStores( params, RequestOptions.create() );
	}


	/**
	 * Get a list of data stores for the specified site.
	 *
	 * @param params
	 *            The parameters required to retrieve a list of data stores
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of DataStoreDto
	 */
	public List<DataStoreDto> getDataStores(
		DataStoreGetListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( DATA_STORES_SITE_URI )
			.responseClass( DataStoreDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Return an empty {@link DataStoreRequestData} object
	 *
	 * @return DataStoreRequestData
	 */
	public DataStoreRequestData newDataStoreRequestData()
	{
		return new DataStoreRequestData();
	}
}