/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.customfield;

import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.customfield.data.CustomFieldRequestData;
import io.mambo.sdk.service.customfield.model.CustomFieldDto;
import io.mambo.sdk.service.customfield.param.CustomFieldCloneParams;
import io.mambo.sdk.service.customfield.param.CustomFieldCreateParams;
import io.mambo.sdk.service.customfield.param.CustomFieldDeleteParams;
import io.mambo.sdk.service.customfield.param.CustomFieldGetListParams;
import io.mambo.sdk.service.customfield.param.CustomFieldGetParams;
import io.mambo.sdk.service.customfield.param.CustomFieldUpdateParams;

public class CustomFieldsService extends AbstractService
{
	private static final String CUSTOM_FIELDS_URI = "/v1/custom_fields";
	private static final String CUSTOM_FIELDS_ID_URI = CUSTOM_FIELDS_URI + "/{customFieldId}";
	private static final String CUSTOM_FIELDS_SITE_URI = "/v1/{siteUrl}/custom_fields";
	private static final String CUSTOM_FIELDS_CLONE_URI = CUSTOM_FIELDS_ID_URI + "/clone";


	public CustomFieldsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * This method is used to create a new custom field.
	 *
	 * @see CustomFieldRequestData
	 *
	 * @param siteUrl
	 *            The site to which the custom field belongs to
	 * @param data
	 *            The custom field request data
	 * @return
	 */
	public CustomFieldDto create( String siteUrl, CustomFieldRequestData data )
	{
		return create( CustomFieldCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * This method is used to create a new custom field.
	 *
	 * @see CustomFieldRequestData
	 *
	 * @param siteUrl
	 *            The site to which the custom field belongs to
	 * @param data
	 *            The custom field request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CustomFieldDto create(
		String siteUrl,
		CustomFieldRequestData data,
		RequestOptions requestOptions )
	{
		return create( CustomFieldCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * This method is used to create a new custom field.
	 *
	 * @param params
	 *            The parameters required to create the custom field
	 * @param data
	 *            The custom field request data
	 * @return
	 */
	public CustomFieldDto create(
		CustomFieldCreateParams params,
		CustomFieldRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to create a new custom field.
	 *
	 * @param params
	 *            The parameters required to create the custom field
	 * @param data
	 *            The custom field request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CustomFieldDto create(
		CustomFieldCreateParams params,
		CustomFieldRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( CUSTOM_FIELDS_SITE_URI )
			.responseClass( CustomFieldDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing custom field by ID.
	 *
	 * @param customFieldId
	 *            The ID of the custom field to update
	 * @param data
	 *            The data with which to update the specified custom field object
	 * @return
	 */
	public CustomFieldDto update( String customFieldId, CustomFieldRequestData data )
	{
		return update( CustomFieldUpdateParams.builder()
			.customFieldId( customFieldId )
			.build(), data );
	}


	/**
	 * Update an existing custom field by ID.
	 *
	 * @param params
	 *            The parameters required to update the custom field
	 * @param data
	 *            The data with which to update the specified custom field object
	 * @return
	 */
	public CustomFieldDto update( CustomFieldUpdateParams params, CustomFieldRequestData data )
	{
	    return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing custom field by ID.
	 *
	 * @param params
	 *            The parameters required to update the custom field
	 * @param data
	 *            The data with which to update the specified custom field object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CustomFieldDto update(
		CustomFieldUpdateParams params,
		CustomFieldRequestData data,
		RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_ID_URI )
	        .responseClass( CustomFieldDto.class )
	        .method( HttpMethod.PUT )
	        .requestData( data )
	        .options( requestOptions )
	        .params( params )
	        .build() );
	}


	/**
	 * Clone a custom field
	 *
	 * @param customFieldId
	 *            The ID of the custom field to clone
	 * @return
	 */
	public CustomFieldDto clone( String customFieldId )
	{
		return clone( CustomFieldCloneParams.builder()
			.customFieldId( customFieldId )
			.build() );
	}


	/**
	 * Clone a custom field
	 *
	 * @param params
	 *            The parameters required to clone the custom field
	 * @return
	 */
	public CustomFieldDto clone( CustomFieldCloneParams params )
	{
	    return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a custom field
	 *
	 * @param params
	 *            The parameters required to clone the custom field
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CustomFieldDto clone(
		CustomFieldCloneParams params,
		RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_CLONE_URI )
	        .responseClass( CustomFieldDto.class )
	        .method( HttpMethod.POST )
	        .options( requestOptions )
	        .params( params )
	        .build() );
	}


	/**
	 * Delete a custom field by ID.
	 *
	 * @param customFieldId
	 *            The ID of the custom field to delete
	 * @return
	 */
	public Status delete( String customFieldId )
	{
		return delete( CustomFieldDeleteParams.builder()
			.customFieldId( customFieldId )
			.build() );
	}


	/**
	 * Delete a custom field by ID.
	 *
	 * @param params
	 *            The parameters required to delete the custom field
	 * @return
	 */
	public Status delete( CustomFieldDeleteParams params )
	{
	    return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a custom field by ID.
	 *
	 * @param params
	 *            The parameters required to delete the custom field
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status delete( CustomFieldDeleteParams params, RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_ID_URI )
	        .responseClass( Status.class )
	        .method( HttpMethod.DELETE )
	        .options( requestOptions )
	        .params( params )
	        .build() );
	}


	/**
	 * Get a custom field by ID
	 *
	 * @param customFieldId
	 *            The ID of the custom field to retrieve
	 * @return
	 */
	public CustomFieldDto get( String customFieldId )
	{
		return get( CustomFieldGetParams.builder()
			.customFieldId( customFieldId )
			.build() );
	}


	/**
	 * Get a custom field by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the custom field
	 * @return
	 */
	public CustomFieldDto get( CustomFieldGetParams params )
	{
	    return get( params, RequestOptions.create() );
	}


	/**
	 * Get a custom field by ID
	 *
	 * @param params
	 *            The parameters required to retrieve the custom field
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CustomFieldDto get( CustomFieldGetParams params, RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_ID_URI )
	        .responseClass( CustomFieldDto.class )
	        .method( HttpMethod.GET )
	        .options( requestOptions )
	        .params( params )
	        .build() );
	}


	/**
	 * Delete a list of custom fields by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @return
	 */
	public Status deleteCustomFields( DeleteRequestData data )
	{
	    return deleteCustomFields( data, RequestOptions.create() );
	}


	/**
	 * Delete a list of custom fields by ID
	 *
	 * @param data
	 *            The {@link DeleteRequestData}
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public Status deleteCustomFields( DeleteRequestData data, RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_URI )
	        .responseClass( Status.class )
	        .method( HttpMethod.DELETE )
	        .requestData( data )
	        .options( requestOptions )
	        .build() );
	}


	/**
	 * Get the list of custom fields for the specified site
	 *
	 * @param siteUrl
	 *            The URL of site containing the siteUrl to use
	 * @return
	 */
	public List<CustomFieldDto> getCustomFields( String siteUrl )
	{
		return getCustomFields( CustomFieldGetListParams.builder()
			.siteUrl( siteUrl )
			.build() );
	}


	/**
	 * Get the list of custom fields for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of custom fields
	 * @return
	 */
	public List<CustomFieldDto> getCustomFields( CustomFieldGetListParams params )
	{
	    return getCustomFields( params, RequestOptions.create() );
	}


	/**
	 * Get the list of custom fields for the specified site
	 *
	 * @param params
	 *            The parameters required to retrieve the list of custom fields
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public List<CustomFieldDto> getCustomFields(
		CustomFieldGetListParams params,
		RequestOptions requestOptions )
	{
	    return apiClient().request( ApiRequest.builder()
	        .apiPath( CUSTOM_FIELDS_SITE_URI )
	        .responseClass( CustomFieldDto.class )
	        .responseType( ResponseType.LIST )
	        .method( HttpMethod.GET )
	        .options( requestOptions )
	        .params( params )
	        .build() );
	}


	/**
	 * Return an empty {@link CustomFieldRequestData} object
	 *
	 * @return
	 */
	public CustomFieldRequestData newCustomFieldRequestData()
	{
		return new CustomFieldRequestData();
	}
}