/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.coupon.param;

import static io.mambo.sdk.utils.Preconditions.checkContainsKey;
import static io.mambo.sdk.utils.Preconditions.checkNotNull;

import java.util.List;

import io.mambo.sdk.service.common.param.AbstractServiceParamsBuilder;
import io.mambo.sdk.service.common.param.ServiceParams;
import io.mambo.sdk.utils.Multimap;

/**
 * Parameters for retrieving coupon states
 */
public class CouponGetStateListParams implements ServiceParams
{
	private Multimap<String, String> params = new Multimap<>();


	private CouponGetStateListParams( Builder builder ) {
		this.params = builder.params();
	}


	public static Builder builder()
	{
		return new Builder();
	}


	@Override
	public Multimap<String, String> asMap()
	{
		return params;
	}


	public static class Builder extends AbstractServiceParamsBuilder<Builder>
	{
		/**
		 * The site from which to retrieve the list of coupon states
		 */
		public Builder siteUrl( String siteUrl )
		{
			checkNotNull( siteUrl, "siteUrl must not be null" );
			params.put( "siteUrl", siteUrl );
			return this;
		}


		/**
		 * Specifies the page of results to retrieve.
		 */
		public Builder page( int page )
		{
			params.put( "page", String.valueOf( page ) );
			return this;
		}


		/**
		 * Specifies the number of results to retrieve, up to a maximum of 100. Check
		 * the documentation to see which APIs support this option
		 */
		public Builder count( int count )
		{
			params.put( "count", String.valueOf( count ) );
			return this;
		}


		/**
		 * The list of tags to filter by (if any)
		 */
		public Builder tags( List<String> tags )
		{
			params.putAll( "tags", tags );
			return this;
		}


		/**
		 * Whether the tags should return items containing any one of the tags or
		 * whether the tags should return only items containing all of the tags.
		 * Possible values: hasAnyOf / hasAllOf Default: hasAnyOf
		 */
		public Builder tagsJoin( String tagsJoin )
		{
			params.put( "tagsJoin", tagsJoin );
			return this;
		}


		/**
		 * The tagUuid to use to filter the list by personalization tags
		 */
		public Builder tagUuid( String tagUuid )
		{
			params.put( "tagUuid", tagUuid );
			return this;
		}


		/**
		 * Specifies the start date of the date range filter
		 */
		public Builder startDate( String startDate )
		{
			params.put( "startDate", startDate );
			return this;
		}


		/**
		 * Specifies the end date of the date range filter
		 */
		public Builder endDate( String endDate )
		{
			params.put( "endDate", endDate );
			return this;
		}


		/**
		 * Filter by awarded user IDs
		 */
		public Builder awardedToUserIds( List<String> awardedToUserIds )
		{
			params.putAll( "awardedToUserIds", awardedToUserIds );
			return this;
		}


		/**
		 * Filter by awarded user UUIDs
		 */
		public Builder awardedToUuids( List<String> awardedToUuids )
		{
			params.putAll( "awardedToUuids", awardedToUuids );
			return this;
		}


		/**
		 * Filter by coupon IDs
		 */
		public Builder couponIds( List<String> couponIds )
		{
			params.putAll( "couponIds", couponIds );
			return this;
		}


		/**
		 * Filter by activity IDs
		 */
		public Builder activityIds( List<String> activityIds )
		{
			params.putAll( "activityIds", activityIds );
			return this;
		}


		/**
		 * Specifies whether only redeemable coupons should be returned (i.e. not redeemed or expired).
		 */
		public Builder withRedeemableOnly( boolean withRedeemableOnly )
		{
			params.put( "withRedeemableOnly", String.valueOf( withRedeemableOnly ) );
			return this;
		}


		/**
		 * Specifies whether only redeemed coupons should be returned.
		 */
		public Builder withRedeemedOnly( boolean withRedeemedOnly )
		{
			params.put( "withRedeemedOnly", String.valueOf( withRedeemedOnly ) );
			return this;
		}


		/**
		 * Specifies whether only expired coupons should be returned.
		 */
		public Builder withExpiredOnly( boolean withExpiredOnly )
		{
			params.put( "withExpiredOnly", String.valueOf( withExpiredOnly ) );
			return this;
		}


		public CouponGetStateListParams build()
		{
			checkContainsKey( params, "siteUrl", "siteUrl must not be null" );
			return new CouponGetStateListParams( this );
		}


		private Multimap<String, String> params()
		{
			return params;
		}
	}
}