/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.coupon.model;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonProperty;

import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.ScarcityDto;
import io.mambo.sdk.service.common.model.interfaces.HasCustomFields;
import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasRecurrence;
import io.mambo.sdk.service.common.model.interfaces.HasTranslation;
import io.mambo.sdk.service.common.model.recurrence.FixedDateRecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.NeverRecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.RecurrenceDto;
import io.mambo.sdk.service.common.model.recurrence.VariablePeriodRecurrenceDto;
import io.mambo.sdk.service.coupon.data.CouponRequestData;
import io.mambo.sdk.service.coupon.model.code.CouponCodeDto;
import io.mambo.sdk.service.coupon.model.code.MultipleCouponCodeDto;
import io.mambo.sdk.service.coupon.model.code.SingleCouponCodeDto;
import io.mambo.sdk.utils.ListUtils;
import io.mambo.sdk.utils.StringUtils;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Coupon data shared by {@link CouponDto} and {@link CouponRequestData}
 */
@ToString
@EqualsAndHashCode
public abstract class AbstractCoupon implements HasTranslation, HasCustomFields, HasRecurrence, HasInitializedData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String name;
	private String type;
	private String helpMessage;
	private String custom;
	private String usage;
	private CouponCodeDto couponCode;
	private Double amount;
	private Boolean active;
	private ScarcityDto scarcity;
	private BuyCriteriaDto buyCriteria;
	private RecurrenceDto expiration;
	private List<CustomFieldValueDto> customFields;
	private List<CouponTranslationDto> translations;

	/**
	 * The name of the coupon.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) {
		initializedFields.add( "name" );
		this.name = name;
	}

	/**
	 * The discount coupon code to be used.
	 * @return
	 */
	public CouponCodeDto getCouponCode() { return couponCode; }
	public void setCouponCode( CouponCodeDto code ) {
		initializedFields.add( "couponCode" );
		couponCode = code;
	}

	public boolean isSingleCouponCode() {
		return StringUtils.equals( "single", getCouponCode().getType() );
	}

	public SingleCouponCodeDto getSingleCouponCode() {
		return (SingleCouponCodeDto) getCouponCode();
	}

	public boolean isMultipleCouponCode() {
		return StringUtils.equals( "multiple", getCouponCode().getType() );
	}

	public MultipleCouponCodeDto getMultipleCouponCode() {
		return (MultipleCouponCodeDto) getCouponCode();
	}

	/**
	 * The type of coupon. Valid values include: percent, fixed, shipping and custom.
	 * @return
	 */
	public String getType() { return type; }
	public void setType( String type ) {
		initializedFields.add( "type" );
		this.type = type;
	}

	/**
	 * This flag indicates whether this coupon is active.
	 * @return
	 */
	public Boolean getActive() { return active; }
	public void setActive( Boolean active ) {
		initializedFields.add( "active" );
		this.active = active;
	}

	/**
	 * In the case of a percent or fixed based coupon, the amount should contain
	 * the percentage or currency amount that the coupon provides.
	 * @return
	 */
	public Double getAmount() { return amount; }
	public void setAmount( Double amount ) {
		initializedFields.add( "amount" );
		this.amount = amount;
	}

	/**
	 * In the case of a custom based coupon, thus field should contain information
	 * relating to the nature of the coupon that is being given. For example: buy 1
	 * get 1 free.
	 * @return
	 */
	public String getCustomPrize() { return custom; }
	@JsonProperty( "custom" )
	public void setCustomPrize( String custom ) {
		initializedFields.add( "custom" );
		this.custom = custom;
	}

	/**
	 * The buy criteria defines whether the coupon can be bought with
	 * points and how the buying process works.
	 * If the coupon cannot be bought then this value should be
	 * set to null.
	 *
	 */
	public BuyCriteriaDto getBuyCriteria() { return buyCriteria; }
	public void setBuyCriteria( BuyCriteriaDto buyCriteria ) {
		initializedFields.add( "buyCriteria" );
		this.buyCriteria = buyCriteria;
	}

	/**
	 * The scarcity represents the overall number of times that a particular
	 * coupon can be unlocked. This effectively creates a limited series
	 * coupon which can only be unlocked X times.
	 * @return
	 */
	public ScarcityDto getScarcity() { return scarcity; }
	public void setScarcity( ScarcityDto scarcity ) {
		initializedFields.add( "scarcity" );
		this.scarcity = scarcity;
	}

	/**
	 * The message to be displayed in the coupon purchasing screen. This can provide
	 * more information about the coupon, how to redeem it, any exceptions for the
	 * coupon, etc.
	 * @return
	 */
	public String getHelpMessage() { return helpMessage; }
	public void setHelpMessage( String helpMessage ) {
		initializedFields.add( "helpMessage" );
		this.helpMessage = helpMessage;
	}

	/**
	 * The usage indicates how many times this coupon can be used. Valid values
	 * include: single or multi. If single is used then the coupon can only be used
	 * once. If multi is used then the coupon can be re-used multiple times (if an
	 * expire period is set, then it can be used re-used until it's expired).
	 *
	 * Important: users can be awarded with the same single use coupon more than
	 * once. This will award the same coupon twice to the user and they should be
	 * able to redeem BOTH those rewards.
	 *
	 * @return
	 */
	public String getUsage() { return usage; }
	public void setUsage( String usage ) {
		initializedFields.add( "usage" );
		this.usage = usage;
	}

	/**
	 * Defines whether this coupon expires or not. If it does, then the relevant time
	 * frame needs to be set.
	 * See the {@link NeverRecurrenceDto}, {@link FixedDateRecurrenceDto} and
	 * {@link VariablePeriodRecurrenceDto} objects for more information.
	 * @return
	 */
	@Override
	public RecurrenceDto getExpiration() { return expiration; }
	@Override
	public void setExpiration( RecurrenceDto expiration ) {
		initializedFields.add( "expiration" );
		this.expiration = expiration;
	}

	/**
	 * Custom fields defined for the coupon. These can contain additional
	 * data or any kind of information you would like to store which isn't a
	 * standard field of the coupon.
	 */
	@Override
	public List<CustomFieldValueDto> getCustomFields() { return customFields; }
	@Override
	public void setCustomFields( List<CustomFieldValueDto> customFields ) {
		initializedFields.add( "customFields" );
		this.customFields = customFields;
	}

	/**
	 * This contains the list of the translations which must be used with the object.
	 * Setting a property directly on the object (such as using setName()) will add
	 * the property to the list of translations using the default language of the server.
	 * If you set a property on the object directly (such as setName()) and then set
	 * the same property using setTranslations(), then the setTranslations() property
	 * will take priority.
	 */
	@Override
	public List<CouponTranslationDto> getTranslations() { return translations; }
	public void setTranslations( List<CouponTranslationDto> translations ) {
		initializedFields.add( "translations" );
		this.translations = translations;
	}
	public void addTranslation( CouponTranslationDto translation ) {
		if( translations == null ) {
			initializedFields.add( "translations" );
			translations = new ArrayList<>();
		}
		translations.add( translation );
	}

	/**
	 * Returns the translation for the languageCode supplied. If no translation
	 * can be found then null will be returned.
	 * @param languageCode
	 * @return
	 */
	public CouponTranslationDto getTranslation( String languageCode )
	{
		if( ListUtils.isEmpty( translations ) ) {
			return null;
		}

		for( CouponTranslationDto translation : translations ) {
			if( StringUtils.equals( languageCode, translation.getLanguageCode() ) ) {
				return translation;
			}
		}

		return null;
	}

	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
