/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.coupon;

import java.io.File;
import java.util.List;

import io.mambo.sdk.http.HttpMethod;
import io.mambo.sdk.http.RequestOptions;
import io.mambo.sdk.http.ResponseType;
import io.mambo.sdk.http.api.ApiRequest;
import io.mambo.sdk.http.api.ApiRequestAdapter;
import io.mambo.sdk.service.common.AbstractService;
import io.mambo.sdk.service.common.data.CustomFieldValueRequestData;
import io.mambo.sdk.service.common.data.DeleteRequestData;
import io.mambo.sdk.service.common.data.FileRequestData;
import io.mambo.sdk.service.common.model.CustomFieldValueDto;
import io.mambo.sdk.service.common.model.response.Status;
import io.mambo.sdk.service.coupon.data.CouponRequestData;
import io.mambo.sdk.service.coupon.data.CouponUserRequestData;
import io.mambo.sdk.service.coupon.model.CouponDto;
import io.mambo.sdk.service.coupon.model.CouponStateDto;
import io.mambo.sdk.service.coupon.param.CouponCloneParams;
import io.mambo.sdk.service.coupon.param.CouponCreateParams;
import io.mambo.sdk.service.coupon.param.CouponCustomFieldParams;
import io.mambo.sdk.service.coupon.param.CouponDeleteParams;
import io.mambo.sdk.service.coupon.param.CouponGetBuyableListParams;
import io.mambo.sdk.service.coupon.param.CouponGetListParams;
import io.mambo.sdk.service.coupon.param.CouponGetParams;
import io.mambo.sdk.service.coupon.param.CouponGetRegularListParams;
import io.mambo.sdk.service.coupon.param.CouponGetStateListParams;
import io.mambo.sdk.service.coupon.param.CouponUpdateParams;
import io.mambo.sdk.service.coupon.param.CouponUploadParams;

/**
 * The CouponsService class handles all Coupon related requests to the Mambo
 * API.
 */
public class CouponsService extends AbstractService
{
	private static final String COUPONS_URI = "/v1/coupons";
	private static final String COUPONS_ID_URI = COUPONS_URI + "/{couponId}";
	private static final String COUPONS_IMAGE_URI = COUPONS_ID_URI + "/image";
	private static final String COUPONS_CLONE_URI = COUPONS_ID_URI + "/clone";
	private static final String COUPONS_CUSTOM_URI = COUPONS_ID_URI + "/custom_fields";
	private static final String COUPONS_CUSTOM_IMAGE_URI = COUPONS_ID_URI + "/custom_fields/image";

	private static final String COUPONS_SITE_URI = "/v1/{siteUrl}/coupons";
	private static final String COUPONS_BUYABLE_URI = COUPONS_SITE_URI + "/buyable";
	private static final String COUPONS_REGULAR_URI = COUPONS_SITE_URI + "/regular";
	private static final String COUPONS_STATE_URI = COUPONS_SITE_URI + "/states";
	private static final String VALIDATE_COUPON_URI = COUPONS_SITE_URI + "/validate";


	public CouponsService( ApiRequestAdapter apiClient ) {
		super( apiClient );
	}


	/**
	 * Create a new coupon.
	 *
	 * @param siteUrl
	 *            The site URL to which the coupon belongs to
	 * @param data
	 *            The coupon request data
	 * @return The created coupon
	 */
	public CouponDto create( String siteUrl, CouponRequestData data )
	{
		return create( CouponCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data );
	}


	/**
	 * Create a new coupon with request options.
	 *
	 * @param siteUrl
	 *            The site URL to which the coupon belongs to
	 * @param data
	 *            The coupon request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The created coupon
	 */
	public CouponDto create(
		String siteUrl,
		CouponRequestData data,
		RequestOptions requestOptions )
	{
		return create( CouponCreateParams.builder()
			.siteUrl( siteUrl )
			.build(), data, requestOptions );
	}


	/**
	 * Create a new coupon.
	 *
	 * @param params
	 *            The parameters required to create the coupon
	 * @param data
	 *            The coupon request data
	 * @return The created coupon
	 */
	public CouponDto create( CouponCreateParams params, CouponRequestData data )
	{
		return create( params, data, RequestOptions.create() );
	}


	/**
	 * Create a new coupon with request options.
	 *
	 * @param params
	 *            The parameters required to create the coupon
	 * @param data
	 *            The coupon request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The created coupon
	 */
	public CouponDto create(
		CouponCreateParams params,
		CouponRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_SITE_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Update an existing coupon by ID.
	 *
	 * @param couponId
	 *            The ID of the coupon to update
	 * @param data
	 *            The data with which to update the specified coupon object
	 * @return The updated coupon
	 */
	public CouponDto update( String couponId, CouponRequestData data )
	{
		return update( CouponUpdateParams.builder()
			.couponId( couponId )
			.build(), data );
	}


	/**
	 * Update an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the coupon
	 * @param data
	 *            The data with which to update the specified coupon object
	 * @return The updated coupon
	 */
	public CouponDto update( CouponUpdateParams params, CouponRequestData data )
	{
		return update( params, data, RequestOptions.create() );
	}


	/**
	 * Update an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the coupon
	 * @param data
	 *            The data with which to update the specified coupon object
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The updated coupon
	 */
	public CouponDto update(
		CouponUpdateParams params,
		CouponRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_ID_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.PUT )
			.requestData( data )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Upload an image for a coupon.
	 *
	 * @param couponId
	 *            The ID of the coupon for which to upload the image
	 * @param image
	 *            The image to upload
	 * @return The updated coupon
	 */
	public CouponDto uploadImage( String couponId, File image )
	{
		return uploadImage( CouponUploadParams.builder()
			.couponId( couponId )
			.build(), image );
	}


	/**
	 * Upload an image for a coupon.
	 *
	 * @param params
	 *            The parameters required to upload the image
	 * @param image
	 *            The image to upload
	 * @return The updated coupon
	 */
	public CouponDto uploadImage( CouponUploadParams params, File image )
	{
		return uploadImage( params, image, RequestOptions.create() );
	}


	/**
	 * Upload an image for a coupon.
	 *
	 * @param params
	 *            The parameters required to upload the image
	 * @param image
	 *            The image to upload
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The updated coupon
	 */
	public CouponDto uploadImage( CouponUploadParams params, File image, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_IMAGE_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.POST )
			.requestData( new FileRequestData( image ) )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Clone a coupon by ID.
	 *
	 * @param couponId
	 *            The ID of the coupon to clone
	 * @return The cloned coupon
	 */
	public CouponDto clone( String couponId )
	{
		return clone( CouponCloneParams.builder().couponId( couponId ).build() );
	}


	/**
	 * Clone a coupon.
	 *
	 * @param params
	 *            The parameters required to clone the coupon
	 * @return The cloned coupon
	 */
	public CouponDto clone( CouponCloneParams params )
	{
		return clone( params, RequestOptions.create() );
	}


	/**
	 * Clone a coupon.
	 *
	 * @param params
	 *            The parameters required to clone the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The cloned coupon
	 */
	public CouponDto clone( CouponCloneParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_CLONE_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.POST )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete a coupon by ID.
	 *
	 * @param couponId
	 *            The ID of the coupon to delete
	 * @return The deletion status
	 */
	public Status delete( String couponId )
	{
		return delete( CouponDeleteParams.builder().couponId( couponId ).build() );
	}


	/**
	 * Delete a coupon.
	 *
	 * @param params
	 *            The parameters required to delete the coupon
	 * @return The deletion status
	 */
	public Status delete( CouponDeleteParams params )
	{
		return delete( params, RequestOptions.create() );
	}


	/**
	 * Delete a coupon.
	 *
	 * @param params
	 *            The parameters required to delete the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The deletion status
	 */
	public Status delete( CouponDeleteParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_ID_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Delete multiple coupons.
	 *
	 * @param data
	 *            The delete request data
	 * @return The deletion status
	 */
	public Status deleteCoupons( DeleteRequestData data )
	{
		return deleteCoupons( data, RequestOptions.create() );
	}


	/**
	 * Delete multiple coupons.
	 *
	 * @param data
	 *            The delete request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The deletion status
	 */
	public Status deleteCoupons( DeleteRequestData data, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_URI )
			.responseClass( Status.class )
			.method( HttpMethod.DELETE )
			.requestData( data )
			.options( requestOptions )
			.build() );
	}


	/**
	 * Get a coupon by ID.
	 *
	 * @param couponId
	 *            The ID of the coupon to retrieve
	 * @return The requested coupon
	 */
	public CouponDto get( String couponId )
	{
		return get( CouponGetParams.builder().couponId( couponId ).build() );
	}


	/**
	 * Get a coupon.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupon
	 * @return The requested coupon
	 */
	public CouponDto get( CouponGetParams params )
	{
		return get( params, RequestOptions.create() );
	}


	/**
	 * Get a coupon.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The requested coupon
	 */
	public CouponDto get( CouponGetParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_ID_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a list of coupons for a site.
	 *
	 * @param siteUrl
	 *            The site containing the coupons
	 * @return List of coupons
	 */
	public List<CouponDto> getCoupons( String siteUrl )
	{
		return getCoupons( CouponGetListParams.builder().siteUrl( siteUrl ).build() );
	}


	/**
	 * Get a list of coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupons
	 * @return List of coupons
	 */
	public List<CouponDto> getCoupons( CouponGetListParams params )
	{
		return getCoupons( params, RequestOptions.create() );
	}


	/**
	 * Get a list of coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupons
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of coupons
	 */
	public List<CouponDto> getCoupons( CouponGetListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_SITE_URI )
			.responseClass( CouponDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a list of buyable coupons for a site.
	 *
	 * @param siteUrl
	 *            The site containing the coupons
	 * @return List of buyable coupons
	 */
	public List<CouponDto> getBuyableCoupons( String siteUrl )
	{
		return getBuyableCoupons( CouponGetBuyableListParams.builder().siteUrl( siteUrl ).build() );
	}


	/**
	 * Get a list of buyable coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the buyable coupons
	 * @return List of buyable coupons
	 */
	public List<CouponDto> getBuyableCoupons( CouponGetBuyableListParams params )
	{
		return getBuyableCoupons( params, RequestOptions.create() );
	}


	/**
	 * Get a list of buyable coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the buyable coupons
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of buyable coupons
	 */
	public List<CouponDto> getBuyableCoupons(
		CouponGetBuyableListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_BUYABLE_URI )
			.responseClass( CouponDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get a list of regular coupons for a site.
	 *
	 * @param siteUrl
	 *            The site containing the coupons
	 * @return List of regular coupons
	 */
	public List<CouponDto> getRegularCoupons( String siteUrl )
	{
		return getRegularCoupons( CouponGetRegularListParams.builder().siteUrl( siteUrl ).build() );
	}


	/**
	 * Get a list of regular coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the regular coupons
	 * @return List of regular coupons
	 */
	public List<CouponDto> getRegularCoupons( CouponGetRegularListParams params )
	{
		return getRegularCoupons( params, RequestOptions.create() );
	}


	/**
	 * Get a list of regular coupons.
	 *
	 * @param params
	 *            The parameters required to retrieve the regular coupons
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of regular coupons
	 */
	public List<CouponDto> getRegularCoupons(
		CouponGetRegularListParams params,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_REGULAR_URI )
			.responseClass( CouponDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Get coupon states for a site.
	 *
	 * @param siteUrl
	 *            The site containing the coupon states
	 * @return List of coupon states
	 */
	public List<CouponStateDto> getCouponStates( String siteUrl )
	{
		return getCouponStates( CouponGetStateListParams.builder().siteUrl( siteUrl ).build() );
	}


	/**
	 * Get coupon states.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupon states
	 * @return List of coupon states
	 */
	public List<CouponStateDto> getCouponStates( CouponGetStateListParams params )
	{
		return getCouponStates( params, RequestOptions.create() );
	}


	/**
	 * Get coupon states.
	 *
	 * @param params
	 *            The parameters required to retrieve the coupon states
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return List of coupon states
	 */
	public List<CouponStateDto> getCouponStates( CouponGetStateListParams params, RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_STATE_URI )
			.responseClass( CouponStateDto.class )
			.responseType( ResponseType.LIST )
			.method( HttpMethod.GET )
			.options( requestOptions )
			.params( params )
			.build() );
	}


	/**
	 * Validate if a coupon can be redeemed by a user.
	 *
	 * @param siteUrl
	 *            The site to which the user and the coupon belong
	 * @param data
	 *            The data required to validate the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The validation status
	 */
	public Status validateRedeem(
		String siteUrl,
		CouponUserRequestData data )
	{
		return validateRedeem( siteUrl, data, RequestOptions.create() );
	}


	/**
	 * Validate if a coupon can be redeemed by a user.
	 *
	 * @param siteUrl
	 *            The site to which the user and the coupon belong
	 * @param data
	 *            The data required to validate the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return The validation status
	 */
	public Status validateRedeem(
		String siteUrl,
		CouponUserRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( VALIDATE_COUPON_URI )
			.responseClass( Status.class )
			.method( HttpMethod.POST )
			.requestData( data )
			.options( requestOptions )
			.params( CouponGetListParams.builder().siteUrl( siteUrl ).build() )
			.build() );
	}


	/**
	 * Create a new empty coupon request data object.
	 *
	 * @return New coupon request data object
	 */
	public CouponRequestData newCouponRequestData()
	{
		return new CouponRequestData();
	}


	/**
	 * This method is used to add custom fields to an existing coupon.
	 *
	 * @param couponId
	 *            The coupon to which to add the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto addCustomFields( String couponId, CustomFieldValueRequestData data )
	{
		CouponCustomFieldParams params = CouponCustomFieldParams.builder().couponId( couponId ).build();
		return modCustomFields( HttpMethod.POST, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to add custom fields to an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto addCustomFields( CouponCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.POST, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to update custom fields of an existing coupon.
	 *
	 * @param couponId
	 *            The coupon to update the custom fields of
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto updateCustomFields( String couponId, CustomFieldValueRequestData data )
	{
		CouponCustomFieldParams params = CouponCustomFieldParams.builder().couponId( couponId ).build();
		return modCustomFields( HttpMethod.PUT, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to update custom fields of an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto updateCustomFields( CouponCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.PUT, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to delete custom fields from an existing coupon.
	 *
	 * @param couponId
	 *            The coupon to delete the custom fields from
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto deleteCustomFields( String couponId, CustomFieldValueRequestData data )
	{
		CouponCustomFieldParams params = CouponCustomFieldParams.builder().couponId( couponId ).build();
		return modCustomFields( HttpMethod.DELETE, params, data, RequestOptions.create() );
	}


	/**
	 * This method is used to delete custom fields from an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @return
	 */
	public CouponDto deleteCustomFields( CouponCustomFieldParams params, CustomFieldValueRequestData data )
	{
		return modCustomFields( HttpMethod.DELETE, params, data, RequestOptions.create() );
	}


	/**
	 * Generic method used to manipulate the custom fields
	 *
	 * @param method
	 *            The HTTP method to use (POST, PUT, DELETE)
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	private CouponDto modCustomFields(
		HttpMethod method,
		CouponCustomFieldParams params,
		CustomFieldValueRequestData data,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_CUSTOM_URI )
			.responseClass( CouponDto.class )
			.method( method )
			.params( params )
			.options( requestOptions )
			.requestData( data )
			.build() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing coupon.
	 *
	 * @param couponId
	 *            The ID of the coupon for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the coupon
	 * @return
	 */
	public CouponDto addCustomField( String couponId, CustomFieldValueDto data, File image )
	{
		return addCustomField( couponId, data, image, RequestOptions.create() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing coupon.
	 *
	 * @param couponId
	 *            The ID of the coupon for which to upload the image
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CouponDto addCustomField(
		String couponId,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		CouponCustomFieldParams params = CouponCustomFieldParams.builder().couponId( couponId ).build();
		return addCustomField( params, data, image, requestOptions );
	}


	/**
	 * This method is used to add a custom field with an image to an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the coupon
	 * @return
	 */
	public CouponDto addCustomField( CouponCustomFieldParams params, CustomFieldValueDto data, File image )
	{
		return addCustomField( params, data, image, RequestOptions.create() );
	}


	/**
	 * This method is used to add a custom field with an image to an existing coupon.
	 *
	 * @param params
	 *            The parameters required to update the custom fields
	 * @param data
	 *            The custom field value request data
	 * @param image
	 *            The image to upload for the coupon
	 * @param requestOptions
	 *            The options to be used with this request
	 * @return
	 */
	public CouponDto addCustomField(
		CouponCustomFieldParams params,
		CustomFieldValueDto data,
		File image,
		RequestOptions requestOptions )
	{
		return apiClient().request( ApiRequest.builder()
			.apiPath( COUPONS_CUSTOM_IMAGE_URI )
			.responseClass( CouponDto.class )
			.method( HttpMethod.POST )
			.params( params )
			.options( requestOptions )
			.requestData( new FileRequestData( image, data ) )
			.build() );
	}
}