/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.response;

import io.mambo.sdk.exception.MamboApiException;
import io.mambo.sdk.exception.MamboExceptionFactory;
import io.mambo.sdk.http.HttpStatus;
import io.mambo.sdk.utils.StringUtils;

/**
 * This is returned when the processing status of an async request is
 * interrogated.
 */
public class ProcessingStatusDto
{
	private static MamboExceptionFactory exceptionFactory = new MamboExceptionFactory();

	private String status;
	private String exceptionMessage;
	private Integer exceptionCode;


	/**
	 * The processing Status of the request
	 *
	 * @return
	 */
	public String getStatus()
	{
		return status;
	}


	public void setStatus( String status )
	{
		this.status = status;
	}


	/**
	 * Returns the exception message when processing the request resulted in an
	 * exception. Returns null if the request was processed successfully or it's
	 * still being processed.
	 *
	 * @return
	 */
	public String getExceptionMessage()
	{
		return exceptionMessage;
	}


	public void setExceptionMessage( String exception )
	{
		this.exceptionMessage = exception;
	}


	/**
	 * Returns the exception status code when processing the request resulted in an
	 * exception. Returns null if the request was processed successfully or it's
	 * still being processed.
	 *
	 * @return
	 */
	public Integer getExceptionCode()
	{
		return exceptionCode;
	}


	public void setExceptionCode( Integer exceptionCode )
	{
		this.exceptionCode = exceptionCode;
	}


	/**
	 * Returns the exception status code when processing the request resulted in an
	 * exception. Returns null if the request was processed successfully or it's
	 * still being processed.
	 *
	 * @return
	 */
	public HttpStatus getExceptionStatus()
	{
		if( exceptionCode == null ){
			return null;
		}
		return HttpStatus.valueOf( exceptionCode );
	}


	/**
	 * Returns a MamboServiceException when processing the request resulted in an
	 * exception. Returns null if the request was processed successfully or it's
	 * still being processed.
	 *
	 * @return
	 */
	public MamboApiException getException()
	{
		if( exceptionMessage == null || exceptionCode == null ) {
			return null;
		}

		return exceptionFactory.createMamboApiException( exceptionMessage, getExceptionStatus() );
	}


	/**
	 * Returns true if the request has not been processed, false otherwise.
	 *
	 * @return
	 */
	public boolean isNotProccessed()
	{
		return StringUtils.equals( "NotProcessed", getStatus() );

	}


	/**
	 * Returns true processing the request resulted in an exception, false
	 * otherwise.
	 *
	 * @return
	 */
	public boolean isFailed()
	{
		return StringUtils.equals( "Failed", getStatus() );
	}


	/**
	 * Returns true if the request was processed successfully, false otherwise.
	 *
	 * @return
	 */
	public boolean isSuccess()
	{
		return StringUtils.equals( "Success", getStatus() );
	}


	/**
	 * Returns true if the request currently being processed, false otherwise.
	 *
	 * @return
	 */
	public boolean isProcessing()
	{
		return StringUtils.equals( "Processing", getStatus() );
	}
}
