/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.awarded;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.common.model.tag.AbstractHasTags;
import io.mambo.sdk.service.reward.model.RewardDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Provides a wrapper around a Reward that has been unlocked
 * and assigned to a User.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class AwardedRewardDto extends AbstractHasTags implements AwardedDto
{
	private String name;
	private String message;
	private String awardedOn;
	private String rewardId;
	private String rewardStatus;
	private RewardDto reward;
	private AwardedPrizeDto prizes;

	/**
	 * The type of awarded object.
	 * @return
	 */
	@Override
	public String getAwardedType() { return "reward"; }

	/**
	 * @see AwardedDto#matchesId(String)
	 */
	@Override
	@JsonIgnore
	public Boolean matchesId( String otherId ) {
		return rewardId.equals( otherId );
	}

	/**
	 * The name of the reward unlocked by the user. See the achievement
	 * or level pages in administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) { this.name = name; }

	/**
	 * The message associated with the reward unlocked by the user.
	 * See the achievement or level pages in administration panel for more information.
	 * @return
	 */
	public String getMessage() { return message; }
	public void setMessage( String message ) { this.message = message; }

	/**
	 * The date on which the reward was unlocked by the user.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	@Override
	public String getAwardedOn() { return awardedOn; }
	public void setAwardedOn( String awardedOn ) { this.awardedOn = awardedOn; }

	/**
	 * The status indicates whether the reward was processed and the
	 * points awarded or not. Possible reward statuses include:
	 * OK, Removed, Rejected
	 * @return
	 */
	public String getRewardStatus() { return rewardStatus; }
	public void setRewardStatus( String rewardStatus ) { this.rewardStatus = rewardStatus; }

	/**
	 * The ID representing the reward that was unlocked. This is used when
	 * retrieving the awarded model through the UsersService getAll*() methods.
	 * @return
	 */
	public String getRewardId() { return rewardId; }
	public void setRewardId( String rewardId ) { this.rewardId = rewardId; }

	/**
	 * The model representing the reward that was unlocked. This will usually
	 * be populated with the underlying Reward, however, if we get this model
	 * from one of the UsersService getAll*() methods, then the rewardId will
	 * be populated instead.
	 * @see RewardDto
	 * @return
	 */
	public RewardDto getReward() { return reward; }
	public void setReward( RewardDto reward ) { this.reward = reward; }

	/**
	 * The prizes object containing the prizes the user earned
	 * by unlocking this reward.
	 * @return
	 */
	public AwardedPrizeDto getPrizes() { return prizes; }
	public void setPrizes( AwardedPrizeDto prizes ) { this.prizes = prizes; }
}
