/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.awarded;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.point.model.PointDto;
import io.mambo.sdk.utils.NumberConverter;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to an object's points.
 */
@ToString
@EqualsAndHashCode
public class AwardedPointDto
{
	private String parentId;
	private String pointId;
	private String expiryOn;
	private Number points;
	private PointDto point;
	private Boolean debited;
	private Boolean rejected;


	/**
	 * The ID of the parent mechanic that awarded the points to the user
	 * @return
	 */
	public String getParentId() { return parentId; }
	public void setParentId( String parentId ) {
		this.parentId = parentId;
	}

	/**
	 * The ID of the points to use with this object.
	 * This field cannot be null.
	 * @return
	 */
	public String getPointId() { return pointId; }
	public void setPointId( String pointId ) {
		this.pointId = pointId;
	}

	/**
	 * The number of points, of the type specified by pointId, associated with the object.
	 * This field cannot be null.
	 */
	public Number getPoints() { return points; }
	public void setPoints( Number points ) {
		this.points = points;
	}

	@JsonIgnore
	public NumberConverter getPointsHelper() { return new NumberConverter( points ); }

	/**
	 * The model representing the underlying point.
	 * @return
	 */
	public PointDto getPoint() { return point; }
	public void setPoint( PointDto point ) {
		this.point = point;
	}

	/**
	 * The date on which the point is going to expire or has expired on.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ<br>
	 * For example: 2013-01-20T20:43:24.094Z
	 * @return
	 */
	public String getExpiryOn() { return expiryOn; }
	public void setExpiryOn( String expiryOn ) {
		this.expiryOn = expiryOn;
	}

	/**
	 * True if the point is expired and the points have
	 * been removed from the relevant user.
	 * @return
	 */
	public Boolean getDebited() { return debited; }
	public void setDebited( Boolean debited ) {
		this.debited = debited;
	}

	/**
	 * True if the point is rejected.
	 * @return
	 */
	public Boolean getRejected() { return rejected; }
	public void setRejected( Boolean rejected ) {
		this.rejected = rejected;
	}
}
