/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.awarded;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.common.model.tag.AbstractHasTags;
import io.mambo.sdk.service.kpi.model.KpiDto;
import io.mambo.sdk.service.reward.model.RewardDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Provides a wrapper around a KPI that has been
 * completed / uncompleted / exceeded / changed by a user.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class AwardedKpiDto extends AbstractHasTags implements AwardedDto
{
	private String name;
	private String targetPeriod;
	private String action;
	private String awardedOn;
	private String kpiId;
	private KpiDto kpi;
	private AwardedPrizeDto prizes;

	/**
	 * The type of awarded object.
	 * @return
	 */
	@Override
	public String getAwardedType() { return "kpi"; }

	/**
	 * @see AwardedDto#matchesId(String)
	 */
	@Override
	@JsonIgnore
	public Boolean matchesId( String otherId ) {
		return kpiId.equals( otherId );
	}

	/**
	 * The name of the KPI awarded to the user. See the KPI
	 * pages in administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) { this.name = name; }

	/**
	 * The date on which the KPI was awarded to the user.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	@Override
	public String getAwardedOn() { return awardedOn; }
	public void setAwardedOn( String awardedOn ) { this.awardedOn = awardedOn; }

	/**
	 * The ID representing the KPI that was awarded.
	 * @return
	 */
	public String getKpiId() { return kpiId; }
	public void setKpiId( String kpiId ) { this.kpiId = kpiId; }

	/**
	 * Returns the target period for which this KPI was awarded
	 * @return
	 */
	public String getTargetPeriod() { return targetPeriod; }
	public void setTargetPeriod( String targetPeriod ) { this.targetPeriod = targetPeriod; }

	/**
	 * Returns the target action which caused this KPI to be awarded
	 * @return
	 */
	public String getAction() { return action; }
	public void setAction( String action ) { this.action = action; }

	/**
	 * The model representing the KPI awarded to the user. This will usually
	 * be populated with the underlying KPI.
	 * @see RewardDto
	 * @return
	 */
	public KpiDto getKpi() { return kpi; }
	public void setKpi( KpiDto kpi ) { this.kpi = kpi; }

	/**
	 * The prizes object containing the prizes the user earned
	 * by performing this behaviour.
	 * @return
	 */
	public AwardedPrizeDto getPrizes() { return prizes; }
	public void setPrizes( AwardedPrizeDto prizes ) { this.prizes = prizes; }
}
