/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.awarded;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.behaviour.model.BehaviourDto;
import io.mambo.sdk.service.common.model.tag.AbstractHasTags;
import io.mambo.sdk.service.reward.model.RewardDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Provides a wrapper around a Behaviour that
 * has been performed by a user.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class AwardedBehaviourDto extends AbstractHasTags implements AwardedDto
{
	private String name;
	private String status;
	private String awardedOn;
	private String behaviourId;
	private String limitExpiresOn;
	private BehaviourDto behaviour;
	private AwardedPrizeDto prizes;

	/**
	 * The type of awarded object.
	 * @return
	 */
	@Override
	public String getAwardedType() { return "behaviour"; }

	/**
	 * @see AwardedDto#matchesId(String)
	 */
	@Override
	@JsonIgnore
	public Boolean matchesId( String otherId ) {
		return behaviourId.equals( otherId );
	}

	/**
	 * The name of the behaviour performed by the user. See the behaviour
	 * pages in administration panel for more information.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) { this.name = name; }

	/**
	 * The date on which the behaviour was performed by the user.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * @return
	 */
	@Override
	public String getAwardedOn() { return awardedOn; }
	public void setAwardedOn( String awardedOn ) { this.awardedOn = awardedOn; }

	/**
	 * The ID representing the behaviour that was performed.
	 * @return
	 */
	public String getBehaviourId() { return behaviourId; }
	public void setBehaviourId( String behaviourId ) { this.behaviourId = behaviourId; }

	/**
	 * The status of the behaviour performed by the user. The status indicates
	 * whether the behaviour was processed and the points awarded or not.
	 * Possible behaviour statuses include: OK, FrequencyException,
	 * CountLimitException, BlacklistedException, NotMatchedException
	 * @return
	 */
	public String getBehaviourStatus() { return status; }
	public void setBehaviourStatus( String status ) { this.status = status; }

	/**
	 * If the behaviour has an expiring rate limit specified, then this
	 * property will contain when the current rate limit will expire
	 * @return
	 */
	public String getLimitExpiresOn() { return limitExpiresOn; }
	public void setLimitExpiresOn( String limitExpiresOn ) { this.limitExpiresOn = limitExpiresOn; }

	/**
	 * The model representing the behaviour performed by the user. This will usually
	 * be populated with the underlying Behaviour.
	 * @see RewardDto
	 * @return
	 */
	public BehaviourDto getBehaviour() { return behaviour; }
	public void setBehaviour( BehaviourDto behaviour ) { this.behaviour = behaviour; }

	/**
	 * The prizes object containing the prizes the user earned
	 * by performing this behaviour.
	 * @return
	 */
	public AwardedPrizeDto getPrizes() { return prizes; }
	public void setPrizes( AwardedPrizeDto prizes ) { this.prizes = prizes; }
}
