/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model.awarded;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnore;

import io.mambo.sdk.service.common.model.PointStoreDto;
import io.mambo.sdk.service.common.model.UserSummaryDto;
import io.mambo.sdk.service.common.model.tag.AbstractHasTags;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Provides a wrapper around an Activity that
 * has been awarded to a user.
 */
@ToString( callSuper = true )
@EqualsAndHashCode( callSuper = true )
public class AwardedActivityDto extends AbstractHasTags implements AwardedDto
{
	private String type;
	private String pointAction;
	private String awardedOn;
	private String activityId;
	private String reason;
	private List<PointStoreDto> points;
	private UserSummaryDto sourceUser;
	private AwardedRewardDto reward;

	/**
	 * The type of awarded object.
	 * @return
	 */
	@Override
	public String getAwardedType() { return "activity"; }

	/**
	 * @see AwardedDto#matchesId(String)
	 */
	@Override
	@JsonIgnore
	public Boolean matchesId( String otherId ) {
		return activityId.equals( otherId );
	}

	/**
	 * Get the ID of the Activity awarded to the user
	 * @return
	 */
	public String getActivityId() { return activityId; }
	public void setActivityId( String activityId ) { this.activityId = activityId; }

	/**
	 * The points associated to the activity awarded to the user.
	 * See the points pages in administration panel for more information.
	 * @return
	 */
	public List<PointStoreDto> getPoints() { return points; }
	public void setPoints( List<PointStoreDto> points ) { this.points = points; }

	/**
	 * The type of activity awarded to the user
	 * @return
	 */
	public String getType() { return type; }
	public void setType( String type ) { this.type = type; }

	/**
	 * If the activity is a point adjustment, then this property
	 * will contain the type of point action being performed
	 * @return
	 */
	public String getPointAction() { return pointAction; }
	public void setPointAction( String pointAction ) { this.pointAction = pointAction; }

	/**
	 * If the activity has a reason associated to it, then this field
	 * contains the reason. This can be used to provide the user with
	 * a more detailed description of the activity.
	 * @return
	 */
	public String getReason() { return reason; }
	public void setReason( String reason ) { this.reason = reason; }

	/**
	 * If the activity is a bounty or gifted, then this property
	 * will contain information about the user who originated the bounty
	 * or the gifting activity.
	 * @return
	 */
	public UserSummaryDto getSourceUser() { return sourceUser; }
	public void setSourceUser( UserSummaryDto sourceUser ) { this.sourceUser= sourceUser ; }

	/**
	 * If the activity is a gift and the user was given a gift
	 * reward, then this will contain the awarded reward's data
	 * @return
	 */
	public AwardedRewardDto getReward() { return reward; }
	public void setReward( AwardedRewardDto reward ) { this.reward= reward ; }

	/**
	 * The date on which the activity was awarded to the user.
	 * This will be a UTC timestamp in ISO 8601 format with
	 * millisecond precision: YYYY-MM-DDTHH:MM:SS.MMMZ
	 * For example: 2013-01-20T20:43:24.094Z
	 * @return
	 */
	@Override
	public String getAwardedOn() { return awardedOn; }
	public void setAwardedOn( String awardedOn ) { this.awardedOn = awardedOn; }
}
