/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.common.model;

import java.util.ArrayList;
import java.util.List;

import io.mambo.sdk.service.common.model.interfaces.HasInitializedData;
import io.mambo.sdk.service.common.model.interfaces.HasPoints;
import io.mambo.sdk.service.coupon.model.CouponDto;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * This class encapsulates data related to prizes
 */
@ToString
@EqualsAndHashCode
public class PrizeDto implements HasPoints, HasInitializedData
{
	protected transient List<String> initializedFields = new ArrayList<>();

	private String couponId;
	private Boolean hideCoupon;
	private CouponDto coupon;
	private List<PointStoreDto> points;
	private PrizeTagsDto addTags;
	private PrizeTagsDto removeTags;


	/**
	 * If this prize has an associated coupon, this should contain
	 * the coupon's ID.
	 * @return
	 */
	public String getCouponId() { return couponId; }
	public void setCouponId( String couponId ) {
		initializedFields.add( "couponId" );
		this.couponId = couponId;
	}

	/**
	 * If this prize has an associated coupon, this determines if
	 * the coupon information should be displayed in the widgets.
	 * If this is set to false then the coupon model should be null.
	 * @return
	 */
	public Boolean getHideCoupon() { return hideCoupon; }
	public void setHideCoupon( Boolean hideCoupon ) {
		initializedFields.add( "hideCoupon" );
		this.hideCoupon = hideCoupon;
	}

	/**
	 * This indicates whether the prize was awarded with a coupon.
	 */
	public Boolean getHasCoupon() { return ( couponId != null ); }

	/**
	 * If a coupon is associated to this prize and it's not hidden, then this will
	 * be populated with the coupon model.
	 * @return
	 */
	public CouponDto getCoupon() { return coupon; }
	public void setCoupon( CouponDto coupon ) {
		this.coupon = coupon;
	}

	/**
	 * The prize points. The points will assigned to a
	 * user who performs a behaviour or unlocks a reward.
	 * @return
	 */
	@Override
	public List<PointStoreDto> getPoints() { return points; }
	@Override
	public void setPoints( List<PointStoreDto> points ) {
		initializedFields.add( "points" );
		this.points = points;
	}
	@Override
	public void addPoint( PointStoreDto point ) {
		initializedFields.add( "points" );
		if( points == null ) {
			points = new ArrayList<>();
		}
		points.add( point );
	}

	/**
	 * Tags to be added to a user when they unlock this prize
	 * @return
	 */
	public PrizeTagsDto getAddTags() { return addTags; }
	public void setAddTags( PrizeTagsDto addTags ) {
		initializedFields.add( "addTags" );
		this.addTags = addTags;
	}

	/**
	 * Tags to be removed from a user when they unlock this prize
	 * @return
	 */
	public PrizeTagsDto getRemoveTags() { return removeTags; }
	public void setRemoveTags( PrizeTagsDto removeTags ) {
		initializedFields.add( "removeTags" );
		this.removeTags = removeTags;
	}


	/**
	 * The list of initialised fields to send to the server
	 */
	@Override
	public List<String> getInitializedFields() {
		return initializedFields;
	}
}
