/*
 * Copyright (C) 2014-2025 Mambo Solutions Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.mambo.sdk.service.behaviour.model.metadata;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * Contains a leaf condition which allows logical operators to be applied
 * to an activities metadata.
 */
@ToString
@EqualsAndHashCode
public class LeafMetadataConditionDto implements MetadataConditionDto
{
	private String name;
	private String value;
	private String operator;


	public LeafMetadataConditionDto() {}

	public LeafMetadataConditionDto( String name, String value, MetadataOperator operator ) {
		this.name = name;
		this.value = value;
		this.operator = operator.value();
	}


	/**
	 * The type of condition: leaf
	 * @return
	 */
	@Override
	public String getType() { return "leaf"; }


	/**
	 * This is the name of the metadata on which we wish to execute the operator.
	 * @return
	 */
	public String getName() { return name; }
	public void setName( String name ) { this.name = name; }


	/**
	 * This is the value of the metadata which will be used with the operator.
	 * @return
	 */
	public String getValue() { return value; }
	public void setValue( String value ) { this.value = value; }


	/**
	 * This is the operator we wish to use when checking the specified metadata.
	 * Valid operators include:
	 * $gt - greaterThan
	 * $gte - greaterThanOrEqualTo
	 * $lt - lessThan
	 * $lte - lessThanOrEqualTo
	 * $on - on
	 * $non - notOn
	 * $after - after
	 * $onafter - onOrAfter
	 * $before - before
	 * $onbefore - onOrBefore
	 * $eq - equalTo
	 * $neq - notEqualTo
	 * $in - contains
	 * $nin - notContains
	 * $sw - startsWith
	 * $nsw - notStartsWith
	 * $ew - endsWith
	 * $new - notEndsWith
	 * @return
	 */
	public String getOperator() { return operator; }

	@JsonProperty( "operator" )
	public void setOperator( String operator ) { this.operator = operator; }

	@JsonIgnore
	public void setOperator( MetadataOperator operator ) { this.operator = operator.value(); }
	public MetadataOperator getMetadataOperator() {
		if( operator == null ) {
			return null;
		}
		return MetadataOperator.fromString( operator );
	}
}
